/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.api.operation.paging;

import static com.mulesoft.connectivity.rest.commons.internal.RestConstants.ATTRIBUTES_VAR;
import static com.mulesoft.connectivity.rest.commons.internal.RestConstants.LINK;
import static com.mulesoft.connectivity.rest.commons.internal.RestConstants.PAYLOAD_VAR;
import static com.mulesoft.connectivity.rest.commons.internal.util.LinkHeaderUtils.buildLinkHeaderMap;
import static com.mulesoft.connectivity.rest.commons.internal.util.RestSdkUtils.isBlank;
import static java.lang.String.format;
import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.mule.runtime.api.metadata.DataType.STRING;

import org.mule.runtime.api.el.BindingContext;
import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;

import com.mulesoft.connectivity.rest.commons.api.connection.RestConnection;
import com.mulesoft.connectivity.rest.commons.api.dw.DWBindings;
import com.mulesoft.connectivity.rest.commons.api.dw.HttpResponseDWBinding;
import com.mulesoft.connectivity.rest.commons.api.operation.HttpResponseAttributes;
import com.mulesoft.connectivity.rest.commons.internal.util.RestRequestBuilder;

import java.util.List;
import java.util.Map;
import java.util.function.Function;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Implementation of {@link RestPagingProvider} for APIs which do paging based on a next URL provided in the response of each
 * request.
 * <p>
 * Request number 1 will obtain the first page plus the next URL embedded in the response. Said URL will be the request number 2,
 * which will reply with the next page and a new URL which overwrites the prior one. The process continues until no new URL is
 * provided or the next page comes back empty.
 *
 * @since 1.0
 */
public class HypermediaPagingProvider extends RestPagingProvider {

  private static final Logger LOGGER = LoggerFactory.getLogger(HypermediaPagingProvider.class);

  private final String nextUrlExpression;

  private boolean firstPage = true;
  private String nextUrl;

  /**
   * Creates a new instance
   *
   * @param nextUrlExpression the DW expression used to extract the next URL from the response
   * @param requestFactory a {@link Function} to generate the request to be used on each page request. Each invocation should
   *        yield a different instance
   * @param expressionLanguage the app's {@link ExpressionLanguage}
   * @param payloadExpression a DW expression to extract the data from the response
   * @param defaultMediaType the {@link MediaType} for the page items if the server doesn't specify one
   * @param responseTimeout the timeout for each request
   */
  public HypermediaPagingProvider(String nextUrlExpression,
                                  Function<RestConnection, RestRequestBuilder> requestFactory,
                                  ExpressionLanguage expressionLanguage,
                                  String payloadExpression,
                                  MediaType defaultMediaType,
                                  Map<String, Object> parameterBindings,
                                  Map<String, Object> customParameterBindings,
                                  int responseTimeout) {
    super(requestFactory, expressionLanguage, payloadExpression,
          defaultMediaType, parameterBindings, customParameterBindings, responseTimeout);
    this.nextUrlExpression = nextUrlExpression;
  }

  @Override
  // TODO: W-12760314 - Testing pending
  public void cleanPagingAttributes() {
    super.cleanPagingAttributes();
    firstPage = true;
  }

  @Override
  protected void configureRequest(RestRequestBuilder requestBuilder) {
    if (!firstPage) {
      if (isBlank(nextUrl)) {
        stopPaging();
        return;
      }

      if (nextUrl.toLowerCase().startsWith("http")) {
        requestBuilder.clearQueryParams();
        requestBuilder.setFullUri(nextUrl);
      } else {
        requestBuilder.setPath(nextUrl);
      }
    }
  }

  @Override
  protected void onPage(List<TypedValue<String>> page, TypedValue<String> rawPage, HttpResponseAttributes responseAttributes) {
    firstPage = false;

    if (isBlank(rawPage.getValue())) {
      nextUrl = null;
      stopPaging();
    } else {
      nextUrl = extractNextUrl(rawPage, responseAttributes);
    }
  }

  private String extractNextUrl(TypedValue<String> rawPage, HttpResponseAttributes responseAttributes) {
    try {
      return (String) evaluate(rawPage, nextUrlExpression, STRING, responseAttributes).getValue();
    } catch (Exception e) {
      throw new MuleRuntimeException(createStaticMessage(
                                                         format("Failed to extract nextUrl from expression: %s",
                                                                nextUrlExpression)),
                                     e);
    }
  }

  @Override
  protected BindingContext toBindingContext(TypedValue<String> content, HttpResponseAttributes httpResponseAttributes) {
    BindingContext.Builder builder = BindingContext.builder()
        .addBinding(PAYLOAD_VAR, content);
    HttpResponseDWBinding httpResponseBinding =
        new HttpResponseDWBinding(content, httpResponseAttributes);
    builder.addBinding(DWBindings.RESPONSE.getBinding(), TypedValue.of(httpResponseBinding));

    if (httpResponseAttributes != null) {
      builder.addBinding(ATTRIBUTES_VAR, TypedValue.of(httpResponseAttributes));
      try {
        String linkHeader = httpResponseAttributes.getHeaders().get(LINK);
        builder.addBinding(LINK, TypedValue.of(buildLinkHeaderMap(linkHeader)));
      } catch (Exception e) {
        LOGGER.warn("Failed to parse Link header", e);
      }
    }

    return builder.build();
  }

}
