/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.internal.interception.expression;

import static java.util.Objects.requireNonNull;

import org.mule.runtime.http.api.domain.message.response.HttpResponse;

import com.mulesoft.connectivity.rest.commons.api.interception.HttpRequest;
import com.mulesoft.connectivity.rest.commons.api.interception.HttpResponseInterceptor;

/**
 * A {@link HttpResponseInterceptor} implementation that applies a precondition and if satisfied and the delegate matches too it
 * delegates the execution to a nested {@link HttpResponseInterceptor}.
 * <p/>
 * This is useful when grouping {@link HttpResponseInterceptor}s by a precondition like: statusCode == 200
 *
 * @since 1.0
 */
public class PreconditionHttpResponseInterceptorDelegate implements HttpResponseInterceptor {

  private Matcher precondition;
  private HttpResponseInterceptor delegate;

  public PreconditionHttpResponseInterceptorDelegate(Matcher precondition,
                                                     HttpResponseInterceptor delegate) {
    requireNonNull(precondition);
    requireNonNull(delegate);

    this.precondition = precondition;
    this.delegate = delegate;
  }

  /**
   * Defines a predicate condition.
   */
  public interface Matcher {

    boolean match(int statusCode);
  }

  @Override
  public HttpResponse intercept(HttpRequest httpRequest, HttpResponse httpResponse) {
    if (!precondition.match(httpResponse.getStatusCode())) {
      return httpResponse;
    }
    return delegate.intercept(httpRequest, httpResponse);
  }

}
