/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.commons.internal.util;

import static org.mule.runtime.api.util.IOUtils.toByteArray;
import static org.mule.runtime.core.api.util.ClassUtils.instantiateClass;
import static org.mule.runtime.core.api.util.StreamingUtils.asCursorProvider;
import static org.slf4j.LoggerFactory.getLogger;

import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.streaming.CursorProvider;
import org.mule.runtime.api.streaming.bytes.CursorStream;

import java.io.IOException;
import java.io.InputStream;
import java.util.LinkedHashMap;
import java.util.Map;

import org.slf4j.Logger;

public class StreamUtils {

  private static final Logger LOGGER = getLogger(StreamUtils.class);

  /**
   * Creates a new {@link Map} where values in entries that are {@link InputStream} or {@link TypedValue<InputStream>} will be
   * replaced with {@link org.mule.runtime.api.streaming.CursorProvider}. Take into account that these streams will be loaded in
   * memory as {@code byte[]} and wrapped with a {@link org.mule.runtime.api.streaming.CursorProvider} to allow accessing it to be
   * read multiple times by honoring the Mule's streaming API.
   *
   * If a {@link org.mule.runtime.api.streaming.Cursor} is present as a value it will get its provider if this one is not closed.
   * 
   * @param map to be transformed/replaced by a {@link org.mule.runtime.api.streaming.CursorProvider}
   * @param <K> the type of the key.
   * @return a new instance with values replaced by a {@link org.mule.runtime.api.streaming.CursorProvider} for those entries that
   *         have a stream value.
   */
  public static <K> Map<K, Object> resolveCursorProvider(Map<K, Object> map) {
    Map<K, Object> transformed;
    try {
      transformed = (Map) instantiateClass(map.getClass(), new Object[0]);
    } catch (Exception e) {
      transformed = new LinkedHashMap();
    }

    for (Map.Entry<K, Object> entry : map.entrySet()) {
      Object value = entry.getValue();
      transformed.put(entry.getKey(), resolveCursorProvider(value));
    }
    return transformed;
  }

  /**
   * If value passed is a {@link InputStream} or {@link TypedValue<InputStream>} it will return a
   * {@link org.mule.runtime.api.streaming.CursorProvider}. Take into account that the stream will be loaded in memory as
   * {@code byte[]} and wrapped with a {@link org.mule.runtime.api.streaming.CursorProvider} to allow accessing it to be read
   * multiple times by honoring the Mule's streaming API.
   *
   * If a {@link org.mule.runtime.api.streaming.Cursor} is passed as parameter it will get its provider if this one is not closed.
   * 
   * @param value to be transformed/replaced by a {@link org.mule.runtime.api.streaming.CursorProvider}
   * @return a {@link org.mule.runtime.api.streaming.CursorProvider} or a
   *         {@link TypedValue<org.mule.runtime.api.streaming.CursorProvider>}.
   */
  public static Object resolveCursorProvider(Object value) {
    if (value instanceof CursorStream && !((CursorStream) value).getProvider().isClosed()) {
      CursorStream cursorStream = (CursorStream) value;
      CursorProvider cursorProvider = cursorStream.getProvider();
      try {
        cursorStream.close();
      } catch (IOException e) {
        LOGGER.error("Error closing cursor stream", e);
      }
      return cursorProvider;
    } else {
      if (value instanceof InputStream) {
        return asCursorProvider(toByteArray((InputStream) value));
      } else if (value instanceof TypedValue) {
        return resolveTypedValueCursorProvider((TypedValue) value);
      } else {
        return value;
      }
    }
  }

  private static TypedValue resolveTypedValueCursorProvider(TypedValue value) {
    Object resolvedValue = resolveCursorProvider(value.getValue());
    return resolvedValue != value.getValue() ? new TypedValue(resolvedValue, value.getDataType()) : value;
  }

}
