/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.HEADER;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.QUERY;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.URI;
import static java.util.Objects.requireNonNull;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

import com.mulesoft.connectivity.rest.sdk.internal.webapi.exception.ModelGenerationException;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorOperation;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.sampledata.SampleDataBuilderUtil;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.trigger.ParameterBinding;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.trigger.ParameterDataType;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.trigger.Trigger;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.trigger.TriggerParameter;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.TypeDefinition;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.TriggerDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.TriggerParameterBindingDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.TriggerParameterBindingsDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.TriggerParameterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.TriggerWatermarkDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.TypeSchemaPool;

import java.util.LinkedList;
import java.util.List;

import javax.ws.rs.core.MediaType;

public class ConnectorTriggerBuilder {

  private final TypeSchemaPool typeSchemaPool;

  public ConnectorTriggerBuilder(TypeSchemaPool typeSchemaPool) {
    this.typeSchemaPool = typeSchemaPool;
  }

  public List<Trigger> buildTriggers(ConnectorDescriptor connectorDescriptor, List<ConnectorOperation> operations)
      throws ModelGenerationException {
    final List<Trigger> triggers = new LinkedList<>();

    if (connectorDescriptor.getTriggers() != null) {
      for (TriggerDescriptor triggerDescriptor : connectorDescriptor.getTriggers()) {
        ConnectorOperation operation = operations.stream()
            .filter(x -> x.getPath().equalsIgnoreCase(triggerDescriptor.getPath()))
            .filter(x -> x.getHttpMethod().equalsIgnoreCase(triggerDescriptor.getMethod().getName()))
            .findFirst().orElse(null);

        requireNonNull(operation);
        triggers.add(buildTrigger(operation, triggerDescriptor));
      }
    }

    return triggers;
  }

  public Trigger buildTrigger(ConnectorOperation operation, TriggerDescriptor triggerDescriptor)
      throws ModelGenerationException {

    TypeDefinition triggerOutputType = buildTriggerOutputType(operation, triggerDescriptor);
    Trigger trigger = new Trigger(triggerDescriptor.getName(),
                                  triggerDescriptor.getDisplayName(),
                                  triggerDescriptor.getDescription(),
                                  triggerDescriptor.getItemsExpression(),
                                  triggerDescriptor.getIdentityExpression(),
                                  getRequestBodyExpression(triggerDescriptor),
                                  getWatermarkExpression(triggerDescriptor),
                                  buildWatermarkType(triggerDescriptor.getWatermark()),
                                  buildParameters(triggerDescriptor.getParameters()),
                                  buildParameterBindings(triggerDescriptor.getParameterBindings()),
                                  triggerOutputType,
                                  operation);
    trigger.setSampleData(buildSampleData(triggerDescriptor));
    return trigger;
  }

  private TypeDefinition buildTriggerOutputType(ConnectorOperation operation, TriggerDescriptor triggerDescriptor)
      throws ModelGenerationException {
    TypeDefinition triggerOutputType;

    if (isNotBlank(triggerDescriptor.getOutputTypeSchema())) {
      ConnectorTypeDefinitionBuilder typeDefinitionBuilder =
          new ConnectorTypeDefinitionBuilder(typeSchemaPool);

      MediaType triggerMediaType;
      if (isNotBlank(triggerDescriptor.getOutputMediaType())) {
        triggerMediaType = MediaType.valueOf(triggerDescriptor.getOutputMediaType());
      } else {
        triggerMediaType = operation.getOutputMetadata().getMediaType();
      }

      triggerOutputType = typeDefinitionBuilder
          .buildTypeDefinition(triggerDescriptor.getOutputTypeSchema(), triggerMediaType);
    } else {
      triggerOutputType = operation.getOutputMetadata();
    }
    return triggerOutputType;
  }

  private String getWatermarkExpression(TriggerDescriptor triggerDescriptor) {
    if (triggerDescriptor == null || triggerDescriptor.getWatermark() == null) {
      return null;
    }

    return triggerDescriptor.getWatermark().getExtraction();
  }

  private String getRequestBodyExpression(TriggerDescriptor triggerDescriptor) {
    if (triggerDescriptor == null || triggerDescriptor.getParameterBindings() == null) {
      return null;
    }

    return triggerDescriptor.getParameterBindings().getRequestBodyExpression();
  }

  private ParameterDataType buildWatermarkType(TriggerWatermarkDescriptor watermarkDescriptor) {
    if (watermarkDescriptor == null || watermarkDescriptor.getDataType() == null) {
      return ParameterDataType.STRING;
    }
    return ParameterDataType.forName(watermarkDescriptor.getDataType().getName());
  }

  private List<ParameterBinding> buildParameterBindings(TriggerParameterBindingsDescriptor parameterBindings) {
    if (parameterBindings == null) {
      return null;
    }

    final List<ParameterBinding> parameterBindingList = new LinkedList<>();

    if (parameterBindings.getUriParameters() != null) {
      for (TriggerParameterBindingDescriptor uriParam : parameterBindings.getUriParameters()) {
        parameterBindingList.add(buildParameterBinding(uriParam, URI));
      }
    }

    if (parameterBindings.getQueryParameters() != null) {
      for (TriggerParameterBindingDescriptor queryParam : parameterBindings.getQueryParameters()) {
        parameterBindingList.add(buildParameterBinding(queryParam, QUERY));
      }
    }

    if (parameterBindings.getHeaders() != null) {
      for (TriggerParameterBindingDescriptor header : parameterBindings.getHeaders()) {
        parameterBindingList.add(buildParameterBinding(header, HEADER));
      }
    }

    return parameterBindingList;
  }

  private ParameterBinding buildParameterBinding(TriggerParameterBindingDescriptor descriptor,
                                                 ParameterType parameterType) {
    return new ParameterBinding(parameterType, descriptor.getName(), descriptor.getValue());
  }

  private List<TriggerParameter> buildParameters(List<TriggerParameterDescriptor> parameterDescriptors) {
    final List<TriggerParameter> parameters = new LinkedList<>();

    for (TriggerParameterDescriptor parameterDescriptor : parameterDescriptors) {
      parameters.add(buildParameter(parameterDescriptor));
    }

    return parameters;
  }

  private TriggerParameter buildParameter(TriggerParameterDescriptor parameterDescriptor) {
    return new TriggerParameter(
                                parameterDescriptor.getName(),
                                parameterDescriptor.getDisplayName(),
                                ParameterDataType.forName(parameterDescriptor.getDataType().getName()),
                                parameterDescriptor.getDescription(),
                                parameterDescriptor.isRequired());
  }

  private boolean buildSampleData(TriggerDescriptor triggerDescriptor) {
    return SampleDataBuilderUtil.shouldHaveSampleData(triggerDescriptor.getSampleDataDescriptor(),
                                                      triggerDescriptor.getMethod().getName());
  }
}
