/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder;

import static javax.ws.rs.core.MediaType.APPLICATION_JSON_TYPE;
import static javax.ws.rs.core.MediaType.APPLICATION_XML_TYPE;
import static org.apache.commons.lang.StringUtils.isNotBlank;

import com.mulesoft.connectivity.rest.sdk.internal.webapi.exception.ModelGenerationException;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.PartParameter;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.PrimitiveTypeDefinition;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.TypeDefinition;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.schema.TypeSchema;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.schema.XmlTypeSchema;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.PartDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIParameterModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.type.APIPrimitiveTypeModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.type.APITypeModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.TypeSchemaPool;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.type.APIType;

import java.util.ArrayList;
import java.util.List;

import javax.ws.rs.core.MediaType;

public class ConnectorTypeDefinitionBuilder {

  private final ConnectorTypeSchemaBuilder typeSchemaBuilder;
  private final ConnectorParameterBuilder parameterBuilder;

  public ConnectorTypeDefinitionBuilder(TypeSchemaPool typeSchemaPool) {
    this.typeSchemaBuilder = new ConnectorTypeSchemaBuilder(typeSchemaPool);
    this.parameterBuilder = new ConnectorParameterBuilder(this);
  }

  public TypeDefinition buildTypeDefinition(APITypeModel apiTypeModel, List<PartDescriptor> partDescriptors)
      throws ModelGenerationException {
    return buildTypeDefinition(apiTypeModel, partDescriptors, apiTypeModel.getMediaType());
  }

  public TypeDefinition buildTypeDefinition(APITypeModel apiTypeModel, List<PartDescriptor> partDescriptors, MediaType mediaType)
      throws ModelGenerationException {

    TypeDefinition.Builder typeDefinitionBuilder =
        TypeDefinition.builder(mediaType,
                               typeSchemaBuilder.buildTypeSchema(apiTypeModel.getAPITypeSchemaModel()),
                               apiTypeModel.getEnumValues(),
                               apiTypeModel.getExample());

    if (isNotBlank(apiTypeModel.getDisplayName())) {
      typeDefinitionBuilder.withDisplayName(apiTypeModel.getDisplayName());
    }

    if (isNotBlank(apiTypeModel.getDescription())) {
      typeDefinitionBuilder.withDescription(apiTypeModel.getDescription());
    }

    APIType apiType = apiTypeModel.getApiType();

    if (apiType == null) {
      return null;
    }
    switch (apiType) {
      case EMPTY:
        return typeDefinitionBuilder.buildEmptyTypeDefinition();
      case PRIMITIVE:
        return typeDefinitionBuilder.buildPrimitiveTypeDefinition(buildPrimitiveType(apiTypeModel.getPrimitiveTypeModel()));
      case OBJECT_TYPE:
        return typeDefinitionBuilder.buildObjectTypeDefinition();
      case ARRAY:
        MediaType innerMediaType =
            apiTypeModel.getInnerType().getMediaType() != null ? apiTypeModel.getInnerType().getMediaType() : mediaType;
        return typeDefinitionBuilder.buildArrayType(buildTypeDefinition(apiTypeModel.getInnerType(), null, innerMediaType));
      case UNION_TYPE:
        List<TypeDefinition> unionTypes = new ArrayList<>();
        for (APITypeModel typeModel : apiTypeModel.getUnionTypes()) {
          TypeDefinition typeDefinition = buildTypeDefinition(typeModel, null, mediaType);
          unionTypes.add(typeDefinition);
        }
        return typeDefinitionBuilder.buildUnionTypeDefinition(unionTypes);
      case MULTIPART:
        List<PartParameter> parts = new ArrayList<>();
        for (APIParameterModel apiParameterModel : apiTypeModel.getParts()) {
          PartDescriptor partDescriptor = null;
          if (partDescriptors != null) {
            partDescriptor = partDescriptors.stream()
                .filter(x -> x.getPartName().equals(apiParameterModel.getExternalName()))
                .findFirst().orElse(null);
          }
          PartParameter partParameter = parameterBuilder.buildPartParameter(apiParameterModel, partDescriptor);
          parts.add(partParameter);
        }
        return typeDefinitionBuilder.buildMultipartTypeDefinition(parts);
      default:
        throw new IllegalArgumentException("Current type '" + apiType + "' is not supported.");
    }
  }

  public TypeDefinition buildTypeDefinition(String rawSchema, MediaType mediaType) throws ModelGenerationException {
    TypeSchema typeSchema = typeSchemaBuilder.buildTypeSchema(rawSchema);

    if (mediaType == null) {
      mediaType = typeSchema instanceof XmlTypeSchema ? APPLICATION_XML_TYPE : APPLICATION_JSON_TYPE;
    }

    return TypeDefinition.builder(mediaType, typeSchema, null, null).buildObjectTypeDefinition();
  }

  private PrimitiveTypeDefinition.PrimitiveType buildPrimitiveType(APIPrimitiveTypeModel primitiveTypeModel) {

    switch (primitiveTypeModel.getPrimitiveType()) {
      case BOOLEAN:
        return PrimitiveTypeDefinition.PrimitiveType.BOOLEAN;
      case DATE:
        return PrimitiveTypeDefinition.PrimitiveType.DATE;
      case DATE_ONLY:
        return PrimitiveTypeDefinition.PrimitiveType.DATE_ONLY;
      case DATE_TIME_ONLY:
        return PrimitiveTypeDefinition.PrimitiveType.DATE_TIME_ONLY;
      case DATE_TIME:
        return PrimitiveTypeDefinition.PrimitiveType.DATE_TIME;
      case FILE:
        return PrimitiveTypeDefinition.PrimitiveType.FILE;
      case NUMBER:
        return PrimitiveTypeDefinition.PrimitiveType.NUMBER;
      case INTEGER:
        return PrimitiveTypeDefinition.PrimitiveType.INTEGER;
      case STRING:
        return PrimitiveTypeDefinition.PrimitiveType.STRING;
      case TIME_ONLY:
        return PrimitiveTypeDefinition.PrimitiveType.TIME_ONLY;
      default:
        throw new IllegalArgumentException("Primitive Type not supported. Should not reach here.");
    }
  }
}
