/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder;

import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.Protocol.HTTP;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.Protocol.HTTPS;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.QueryParamArrayFormat.MULTIMAP;
import static com.mulesoft.connectivity.rest.sdk.internal.webapi.util.XmlUtils.getXmlName;
import static com.mulesoft.connectivity.rest.sdk.internal.webapi.util.XmlUtils.removeMavenArtifactUnwantedCharacters;
import static com.mulesoft.connectivity.rest.sdk.internal.webapi.util.XmlUtils.removeMavenGroupUnwantedCharacters;
import static com.mulesoft.connectivity.rest.sdk.internal.webapi.util.XmlUtils.removeMavenVersionUnwantedCharacters;

import com.mulesoft.connectivity.rest.sdk.internal.webapi.exception.ModelGenerationException;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorCategory;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorOperation;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorPackage;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.Protocol;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.QueryParamArrayFormat;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.pagination.Pagination;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.uri.BaseUri;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.PaginationDeclarationDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.TypeSchemaPool;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class ConnectorModelBuilder {

  private final static List<Protocol> DEFAULT_PROTOCOLS;

  static {
    DEFAULT_PROTOCOLS = new ArrayList<>();
    DEFAULT_PROTOCOLS.add(HTTP);
    DEFAULT_PROTOCOLS.add(HTTPS);
  }

  private final TypeSchemaPool typeSchemaPool = new TypeSchemaPool();
  private final ConnectorOperationBuilder connectorOperationBuilder = new ConnectorOperationBuilder(typeSchemaPool);
  private final ConnectorTriggerBuilder connectorTriggerBuilder = new ConnectorTriggerBuilder(typeSchemaPool);

  public ConnectorModel buildConnectorModel(APIModel apiModel, ConnectorDescriptor connectorDescriptor)
      throws ModelGenerationException {
    List<Protocol> protocols = buildProtocols(apiModel);
    List<ConnectorOperation> operations = connectorOperationBuilder.buildOperations(apiModel, connectorDescriptor);

    return new ConnectorModel(buildConnectorName(connectorDescriptor),
                              connectorDescriptor.getConnectorDescription(),
                              buildMavenGroup(connectorDescriptor),
                              buildMavenArtifactId(connectorDescriptor),
                              buildMavenVersion(connectorDescriptor),
                              buildBasePackage(connectorDescriptor),
                              buildConnectorCategory(connectorDescriptor),
                              buildBaseUri(connectorDescriptor, apiModel, protocols),
                              protocols,
                              operations.stream().filter(x -> !x.isIgnored()).collect(toList()),
                              buildPaginations(connectorDescriptor),
                              buildExtensionXml(connectorDescriptor),
                              buildSkipOutputTypeValidation(connectorDescriptor),
                              buildQueryParamArrayFormat(connectorDescriptor),
                              connectorTriggerBuilder.buildTriggers(connectorDescriptor, operations));
  }

  protected QueryParamArrayFormat buildQueryParamArrayFormat(ConnectorDescriptor connectorDescriptor) {
    if (isNotBlank(connectorDescriptor.getQueryParamArrayFormat())) {
      return QueryParamArrayFormat.valueOf(connectorDescriptor.getQueryParamArrayFormat().toUpperCase());
    }
    return MULTIMAP;
  }

  private static String buildConnectorName(ConnectorDescriptor connectorDescriptor) {
    String connectorName = connectorDescriptor.getConnectorName().trim();
    if (connectorName.toLowerCase().endsWith("api")) {
      connectorName = connectorName.substring(0, connectorName.lastIndexOf("api"));
    }

    if (connectorName.toLowerCase().endsWith("connector")) {
      return connectorName;
    }

    return connectorName + " Connector";
  }

  private static Boolean buildSkipOutputTypeValidation(ConnectorDescriptor connectorDescriptor) {
    return connectorDescriptor.getSkipOutputTypeValidation();
  }

  private static String buildExtensionXml(ConnectorDescriptor connectorDescriptor) {
    if (isNotBlank(connectorDescriptor.getExtensionXml())) {
      return connectorDescriptor.getExtensionXml();
    }
    return getXmlName(connectorDescriptor.getConnectorName());
  }

  private static String buildBasePackage(ConnectorDescriptor connectorDescriptor) {
    if (isNotBlank(connectorDescriptor.getBaseJavaPackage())) {
      return connectorDescriptor.getBaseJavaPackage();
    }
    return ConnectorPackage.buildBasePackage(connectorDescriptor.getConnectorName());
  }

  private static List<Protocol> buildProtocols(APIModel apiModel) {
    return apiModel.getProtocols().isEmpty() ? DEFAULT_PROTOCOLS
        : apiModel.getProtocols().stream().map(Protocol::getFromString).collect(toList());
  }

  private static String buildMavenGroup(ConnectorDescriptor connectorDescriptor) {
    return removeMavenGroupUnwantedCharacters(connectorDescriptor.getConnectorGav().getGroupId());
  }

  private static String buildMavenArtifactId(ConnectorDescriptor connectorDescriptor) {
    return removeMavenArtifactUnwantedCharacters(connectorDescriptor.getConnectorGav().getArtifactId());
  }

  private static String buildMavenVersion(ConnectorDescriptor connectorDescriptor) {
    return removeMavenVersionUnwantedCharacters(connectorDescriptor.getConnectorGav().getVersion());
  }

  private static ConnectorCategory buildConnectorCategory(ConnectorDescriptor connectorDescriptor) {
    return ConnectorCategory.fromString(connectorDescriptor.getConnectorCategory());
  }

  private static BaseUri buildBaseUri(ConnectorDescriptor connectorDescriptor, APIModel apiModel, List<Protocol> protocols) {

    if (connectorDescriptor.getBaseUri() == null) {
      return new BaseUri(apiModel.getBaseUri(), apiModel.getApiVersion(), protocols);
    }

    BaseUri generatedUri = new BaseUri(connectorDescriptor.getBaseUri().getValue(),
                                       BaseUri.Type.valueOfName(connectorDescriptor.getBaseUri().getType()),
                                       apiModel.getApiVersion(),
                                       protocols);

    connectorDescriptor.getBaseUri().getMapping()
        .forEach(x -> generatedUri.addMultipleBaseUri(x.getName(), x.getValue(), x.isDefault()));

    return generatedUri;
  }

  private static Map<String, Pagination> buildPaginations(ConnectorDescriptor connectorDescriptor) {
    List<Pagination> paginations = new ArrayList<>(connectorDescriptor.getPaginations().size());

    for (PaginationDeclarationDescriptor paginationDescriptor : connectorDescriptor.getPaginations()) {

      paginations.add(new Pagination(paginationDescriptor.getName(),
                                     paginationDescriptor.getType(),
                                     paginationDescriptor.getPaginationParameterName(),
                                     paginationDescriptor.getNextTokenExpression() != null
                                         ? paginationDescriptor.getNextTokenExpression().getExpression() : null,
                                     paginationDescriptor.getInitialOffset() != null
                                         ? paginationDescriptor.getInitialOffset() : null,
                                     paginationDescriptor.getInitialPageNumber() != null
                                         ? paginationDescriptor.getInitialPageNumber() : null,
                                     paginationDescriptor.getPagingResponseExpression() != null
                                         ? paginationDescriptor.getPagingResponseExpression().getExpression() : null));
    }

    Map<String, Pagination> paginationMap = new HashMap<>();
    paginations.forEach(paging -> paginationMap.put(paging.getName(), paging));

    return paginationMap;
  }
}
