/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder;

import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.isBlank;

import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ExpressionDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.exception.ModelGenerationException;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.Parameter;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.ConnectorSecurityScheme;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.NotSupportedScheme;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.TestConnectionConfig;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.TestConnectionValidationConfig;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.UnsecuredScheme;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.SecuritySchemeDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.TestConnectionDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.TestConnectionResponseValidationDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APISecuritySchemeModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APISecuritySchemeType;

import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.http.api.HttpConstants.Method;

import java.util.LinkedList;
import java.util.List;

public class ConnectorSecuritySchemeBuilder {

  private final ConnectorParameterBuilder parameterBuilder;

  public ConnectorSecuritySchemeBuilder(ConnectorTypeDefinitionBuilder connectorTypeDefinitionBuilder) {
    this.parameterBuilder = new ConnectorParameterBuilder(connectorTypeDefinitionBuilder);
  }

  public List<ConnectorSecurityScheme> buildSecuritySchemes(List<APISecuritySchemeModel> apiSecuritySchemeModels,
                                                            ConnectorDescriptor connectorDescriptor)
      throws ModelGenerationException {
    List<SecuritySchemeDescriptor> securityDescriptors = connectorDescriptor.getSecurity();
    List<ConnectorSecurityScheme> apiSecuritySchemes = new LinkedList<>();

    if (!apiSecuritySchemeModels.isEmpty()) {
      for (APISecuritySchemeModel securitySchemeModel : apiSecuritySchemeModels) {
        ConnectorSecurityScheme apiSecurityScheme =
            buildSecurityScheme(securitySchemeModel, getSecuritySchemeDescriptor(securityDescriptors, securitySchemeModel),
                                connectorDescriptor);
        if (apiSecurityScheme == null) {
          continue;
        }
        if (!(apiSecurityScheme instanceof NotSupportedScheme)) {
          apiSecuritySchemes.add(apiSecurityScheme);
        }
      }
    } else {
      apiSecuritySchemes.add(new UnsecuredScheme(buildTestConnectionConfig(connectorDescriptor.getTestConnection())));
    }

    return apiSecuritySchemes;
  }

  private TestConnectionConfig buildTestConnectionConfig(TestConnectionDescriptor testConnectionDescriptor) {
    if (testConnectionDescriptor == null) {
      return null;
    }

    return new TestConnectionConfig(testConnectionDescriptor.getPath(),
                                    testConnectionDescriptor.getMethod() == null ? null
                                        : Method.valueOf(testConnectionDescriptor.getMethod().toUpperCase()),
                                    testConnectionDescriptor.getValidStatusCodes(),
                                    testConnectionDescriptor.getMediaType() == null ? null
                                        : MediaType.parse(testConnectionDescriptor.getMediaType()),
                                    testConnectionDescriptor.getResponseValidation().stream()
                                        .map(this::buildTestConnectionValidationConfig)
                                        .collect(toList()));
  }

  private TestConnectionValidationConfig buildTestConnectionValidationConfig(TestConnectionResponseValidationDescriptor responseValidation) {
    if (responseValidation.getValidationExpression() == null
        || isBlank(responseValidation.getValidationExpression().getExpression())) {
      return null;
    }

    String errorExpression = responseValidation.getErrorTemplateExpression() != null
        ? responseValidation.getErrorTemplateExpression().getExpression()
        : null;

    return new TestConnectionValidationConfig(responseValidation.getValidationExpression().getExpression(), errorExpression);
  }

  private SecuritySchemeDescriptor getSecuritySchemeDescriptor(List<SecuritySchemeDescriptor> securities,
                                                               APISecuritySchemeModel securitySchemeModel) {
    return securities.stream()
        .filter(x -> x.getDisplayName().equalsIgnoreCase(securitySchemeModel.getName()))
        .findFirst().orElse(null);
  }

  private ConnectorSecurityScheme buildSecurityScheme(APISecuritySchemeModel securitySchemeModel,
                                                      SecuritySchemeDescriptor securitySchemeDescriptor,
                                                      ConnectorDescriptor connectorDescriptor)
      throws ModelGenerationException {

    TestConnectionDescriptor testConnectionDescriptor =
        (securitySchemeDescriptor != null && securitySchemeDescriptor.getTestConnection() != null)
            ? securitySchemeDescriptor.getTestConnection()
            : connectorDescriptor.getTestConnection();
    TestConnectionConfig testConnectionConfig = buildTestConnectionConfig(testConnectionDescriptor);

    if (securitySchemeDescriptor != null && securitySchemeDescriptor.isIgnored()) {
      return null;
    }

    APISecuritySchemeType apiSecuritySchemeType = securitySchemeModel.getSecuritySchemeType();
    List<String> collisionNames = new LinkedList<>();

    List<Parameter> customQueryParams =
        parameterBuilder.buildParameterList(securitySchemeModel.getCustomQueryParams(), securitySchemeDescriptor, collisionNames);
    collisionNames.addAll(customQueryParams.stream().map(Parameter::getInternalName).collect(toList()));
    List<Parameter> customHeaders =
        parameterBuilder.buildParameterList(securitySchemeModel.getCustomHeaders(), securitySchemeDescriptor, collisionNames);

    ConnectorSecurityScheme.Builder builder =
        ConnectorSecurityScheme.builder(securitySchemeModel.getName(), customQueryParams, customHeaders, testConnectionConfig);

    switch (apiSecuritySchemeType) {
      case BASIC_AUTH:
        return builder.buildBasicAuthScheme();
      case CUSTOM:
        return builder.buildCustomAuthenticationScheme();
      case DIGEST:
        return builder.buildDigestAuthenticationSchemeScheme();
      case PASS_THROUGH:
        return builder.buildPassThroughScheme();
      case OAUTH2_AUTHORIZATION_CODE:
        return builder.buildOAuth2AuthorizationCodeScheme(securitySchemeModel.getAuthorizationUri(),
                                                          securitySchemeModel.getAccessTokenUri(),
                                                          securitySchemeModel.getScopes(),
                                                          getRefreshTokenConditionExpression(securitySchemeDescriptor));
      case OAUTH2_CLIENT_CREDENTIALS:
        return builder.buildOAuth2ClientCredentialsScheme(securitySchemeModel.getAuthorizationUri(),
                                                          securitySchemeModel.getAccessTokenUri(),
                                                          securitySchemeModel.getScopes(),
                                                          getRefreshTokenConditionExpression(securitySchemeDescriptor));
      case NOT_SUPPORTED:
        return builder.buildUnsecuredScheme();
      default:
        throw new IllegalArgumentException("Current scheme '" + apiSecuritySchemeType + "' is not supported.");
    }
  }

  private String getRefreshTokenConditionExpression(SecuritySchemeDescriptor securitySchemeDescriptor) {
    if (securitySchemeDescriptor == null) {
      return null;
    }
    return securitySchemeDescriptor.getRefreshTokenConditionExpressionDescriptor().map(ExpressionDescriptor::getExpression)
        .orElse(null);
  }
}

