/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.connectormodel.util;

import static com.mulesoft.connectivity.rest.sdk.internal.webapi.util.ParserUtils.splitCaps;
import static java.util.Arrays.stream;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.apache.commons.lang3.StringUtils.join;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.HTTPMethod;

import java.util.LinkedList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.lang3.StringUtils;

public class OperationNamingUtils {

  private static final String URI_PARAMETER_REGEX = "\\{(.*?)}";
  private static final Pattern URI_PARAMETER_PATTERN = Pattern.compile(URI_PARAMETER_REGEX);

  private OperationNamingUtils() {}

  public static List<String> getPathResources(String path) {
    return getPathElements(path, false);
  }

  public static List<String> getPathParameters(String path) {
    return getPathElements(path, true);
  }

  /***
   * Returns a list of the resources or parameters present in the path If parameters = true, it returns all the parameters present
   * in the path If parameters = false, it return all the resources present in the path
   * 
   * @param path The path where to extract the elements from
   * @param findParameters Indicates if you want to extract parameters or resources
   * @return A list with the parameters/resources
   */
  private static List<String> getPathElements(String path, boolean findParameters) {
    List<String> elements = new LinkedList<>();

    String remainingPath = replaceODataParams(path);

    // Collects the names of the parameters or resources.
    while (isNotBlank(remainingPath)) {
      int resourceEndIndex = remainingPath.indexOf('/');

      if (resourceEndIndex < 0) {
        resourceEndIndex = remainingPath.length();
      }

      String pathSegment = remainingPath.substring(0, resourceEndIndex);

      // Appends the parameter/resource to the list.
      if (isNotBlank(pathSegment)) {
        if (findParameters) {
          for (String uriParameter : getUriParametersFromPathSegment(pathSegment)) {
            elements.add(getCapUnderscoreName(uriParameter));
          }
        } else {
          for (String uriResource : getUriResourcesFromPathSegment(pathSegment)) {
            elements.add(getCapUnderscoreName(uriResource));
          }
        }
      }

      if (resourceEndIndex < remainingPath.length()) {
        remainingPath = remainingPath.substring(resourceEndIndex + 1);
      } else {
        remainingPath = remainingPath.substring(resourceEndIndex);
      }
    }

    return elements;
  }

  /**
   * Builds the operation name according to the method and path. GET /towns -> getTowns GET /towns/{id} -> getTownsById POST
   * /towns -> createTowns PUT /towns/{id}/streets -> updateTownsStreetsById PATCH /towns/{id}/streets -> patchTownsStreetsById
   * DELETE /towns/{townId}/streets/{streetId} -> deleteTownsStreetsByTownIdStreetId
   */
  public static String buildCanonicalOperationName(HTTPMethod method, String path) {
    List<String> resources = getPathResources(path);
    List<String> parameters = getPathParameters(path);

    StringBuilder operationNameBuilder = new StringBuilder(method.getActionName());

    if (!resources.isEmpty()) {
      operationNameBuilder.append("_").append(join(resources, "_"));
    }

    if (!parameters.isEmpty()) {
      operationNameBuilder.append("_BY_").append(join(parameters, "_"));
    }

    return operationNameBuilder.toString();
  }

  private static String replaceODataParams(String string) {
    Pattern oDataParameterPattern = Pattern.compile("([^\\s(){}=,]+)='\\{([^\\s(){}=,]+)}'");
    Matcher oDataParameterMatcher = oDataParameterPattern.matcher(string);
    return oDataParameterMatcher.replaceAll("$1");
  }

  private static List<String> getUriParametersFromPathSegment(String pathSegment) {
    Matcher matcher = URI_PARAMETER_PATTERN.matcher(pathSegment);

    List<String> results = new LinkedList<>();

    while (matcher.find()) {
      results.add(matcher.group(1));
    }

    return results;
  }

  private static List<String> getUriResourcesFromPathSegment(String pathSegment) {
    return stream(pathSegment.split(URI_PARAMETER_REGEX))
        .filter(StringUtils::isNotBlank)
        .collect(toList());
  }

  private static String getCapUnderscoreName(String name) {
    return splitCapsWithUnderscores(getSanitizedOperationName(name)).toUpperCase();
  }

  private static String getSanitizedOperationName(String name) {
    name = name.replaceAll("-", "_");
    name = name.replaceAll("[^a-zA-Z0-9_-]", EMPTY);

    if (name.startsWith("_"))
      name = name.replaceFirst("_", EMPTY);

    return name;
  }

  private static String splitCapsWithUnderscores(String name) {
    return splitCaps(name, "_");
  }

}
