/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.connectormodel.uri;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.uri.BaseUri.Type.MULTIPLE;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.uri.BaseUri.Type.PARAMETER;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.Protocol.HTTP;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.Protocol.HTTPS;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.Protocol;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class BaseUri {

  private static final String VERSION_URI_PARAM_REGEX = "\\{version\\}";

  public enum Type {

    FIXED("fixed", "Fixed base URI specified in the API Spec"), PARAMETER("parameter",
        "Parameter base URI, each instance/tenant gets its own"), MULTIPLE("multiple",
            "Multiple base URI, each operation defines its own base URI");

    private static final Map<String, Type> BY_TYPE = new HashMap<>();

    static {
      for (Type e : values()) {
        BY_TYPE.put(e.getName(), e);
      }
    }

    private final String name;
    private final String description;

    Type(String name, String description) {
      this.name = name;
      this.description = description;
    }

    public String getName() {
      return this.name;
    }

    public String getDescription() {
      return this.description;
    }

    public static Type valueOfName(String name) {
      return BY_TYPE.get(name.toLowerCase());
    }
  }


  private static final String PROTOCOL_SEPARATOR = "://";

  private String uri;
  private final Type type;
  private final String apiVersion;
  private final List<Protocol> supportedProtocols;

  private final Map<String, String> uriMappings = new HashMap<>();

  public BaseUri(String uri, String apiVersion, List<Protocol> supportedProtocols) {
    this(uri, PARAMETER, apiVersion, supportedProtocols);
  }

  public BaseUri(String uri, Type type, String apiVersion, List<Protocol> supportedProtocols) {
    this.supportedProtocols = supportedProtocols;
    this.uri = addProtocolToUri(uri, supportedProtocols);
    this.type = type;
    this.apiVersion = apiVersion;
  }

  private String addProtocolToUri(String uri, List<Protocol> supportedProtocols) {
    if (uri == null) {
      return null;
    }

    if (uri.contains(PROTOCOL_SEPARATOR)) {
      return uri;
    }

    if (!supportedProtocols.isEmpty() && !supportedProtocols.contains(HTTPS)) {
      return HTTP + PROTOCOL_SEPARATOR + uri;
    } else {
      return HTTPS + PROTOCOL_SEPARATOR + uri;
    }
  }

  public String getUri() {
    if (isNotBlank(uri)) {
      if (isNotBlank(apiVersion)) {
        uri = uri.replaceAll(VERSION_URI_PARAM_REGEX, apiVersion);
      }

      if (uri.endsWith("/")) {
        uri = uri.substring(0, uri.length() - 1);
      }
    }

    return uri;
  }

  public Type getType() {
    return type;
  }

  public String getApiVersion() {
    return apiVersion;
  }

  public Map<String, String> getUriMappings() {
    return uriMappings;
  }

  public void addMultipleBaseUri(String uriId, String uri, boolean isDefaultUri) {
    this.uriMappings.put(uriId, uri);

    if (isDefaultUri) {
      this.uri = addProtocolToUri(uri, supportedProtocols);
    }
  }

  public String getMultipleBaseUriOrDefault(String uriId) {
    return this.uriMappings.getOrDefault(uriId, uri);
  }

  public boolean isMultipleBaseUri() {
    return type == MULTIPLE;
  }

  public boolean isParameterizedBaseUri() {
    return type == PARAMETER;
  }
}
