/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.QueryParamArrayFormat.MULTIMAP;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorOperation;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.HTTPMethod;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.QueryParamArrayFormat;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.TypeSchemaPool;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.resolver.sampledata.ConnectorSampleDataBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dw.OperationDisplayNameExpressionHandler;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dw.OperationIdentifierExpressionHandler;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.pagination.Pagination;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.Parameter;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.TypeDefinition;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.EndPointDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.PartDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.resolvers.ResolverExpressionDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.sampledata.SampleDataDefinitionDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.exception.ModelGenerationException;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIOperationModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.type.APITypeModel;

import java.util.ArrayList;
import java.util.List;

import javax.ws.rs.core.MediaType;



public class ConnectorOperationBuilder {

  private final ConnectorTypeDefinitionBuilder typeDefinitionBuilder;
  private final ConnectorParameterBuilder parameterBuilder;
  private final ConnectorSecuritySchemeBuilder securitySchemeBuilder;
  private final ConnectorSampleDataBuilder sampleDataBuilder;

  public ConnectorOperationBuilder(TypeSchemaPool typeSchemaPool) {
    this.typeDefinitionBuilder = new ConnectorTypeDefinitionBuilder(typeSchemaPool);
    this.parameterBuilder = new ConnectorParameterBuilder(typeDefinitionBuilder);
    this.securitySchemeBuilder = new ConnectorSecuritySchemeBuilder(typeDefinitionBuilder);
    this.sampleDataBuilder = new ConnectorSampleDataBuilder();
  }

  public List<ConnectorOperation> buildOperations(APIModel apiModel, ConnectorDescriptor connectorDescriptor,
                                                  List<Pagination> paginations,
                                                  OperationIdentifierExpressionHandler operationIdentifierExpressionHandler,
                                                  OperationDisplayNameExpressionHandler operationDisplayNameExpressionHandler)
      throws ModelGenerationException {

    List<ConnectorOperation> operations = new ArrayList<>();
    for (APIOperationModel op : apiModel.getOperationsModel()) {
      operations.add(buildOperation(op,
                                    getEndpointDescriptor(op, connectorDescriptor),
                                    getOperationDescriptor(op, connectorDescriptor),
                                    connectorDescriptor,
                                    paginations, operationIdentifierExpressionHandler, operationDisplayNameExpressionHandler));
    }
    return operations;
  }

  private static EndPointDescriptor getEndpointDescriptor(APIOperationModel operationModel,
                                                          ConnectorDescriptor connectorDescriptor) {
    return connectorDescriptor.getEndpoints().stream()
        .filter(x -> x.getPath().equalsIgnoreCase(operationModel.getPath()))
        .findFirst().orElse(null);
  }

  private static OperationDescriptor getOperationDescriptor(APIOperationModel operationModel,
                                                            ConnectorDescriptor connectorDescriptor) {
    return connectorDescriptor.getEndpoints().stream()
        .filter(x -> x.getPath().equalsIgnoreCase(operationModel.getPath()))
        .flatMap(x -> x.getOperations().stream())
        .filter(x -> x.getMethod().equalsIgnoreCase(operationModel.getHttpMethod()))
        .findFirst().orElse(null);
  }

  public ConnectorOperation buildOperation(APIOperationModel apiOperationModel, EndPointDescriptor endPointDescriptor,
                                           OperationDescriptor operationDescriptor, ConnectorDescriptor connectorDescriptor,
                                           List<Pagination> paginations,
                                           OperationIdentifierExpressionHandler operationIdentifierExpressionHandler,
                                           OperationDisplayNameExpressionHandler operationDisplayNameExpressionHandler)
      throws ModelGenerationException {
    List<Parameter> uriParameters =
        parameterBuilder.buildParameterList(apiOperationModel.getUriParamsModel(), operationDescriptor);
    List<Parameter> queryParameters =
        parameterBuilder.buildParameterList(apiOperationModel.getQueryParamsModel(), operationDescriptor);
    List<Parameter> headers =
        parameterBuilder.buildParameterList(apiOperationModel.getHeadersModel(), operationDescriptor);

    TypeDefinition outputTypeMetadata = buildOutputTypeMetadata(apiOperationModel, operationDescriptor, connectorDescriptor);

    // Accept header will be inferred from the response media type, so we remove it from the operation parameters.
    if (outputTypeMetadata != null && outputTypeMetadata.getMediaType() != null) {
      headers.stream()
          .filter(x -> x.getExternalName().equalsIgnoreCase("accept"))
          .findFirst()
          .ifPresent(headers::remove);
    }

    HTTPMethod method = HTTPMethod.fromString(apiOperationModel.getHttpMethod());

    ConnectorOperation operation =
        new ConnectorOperation(buildOperationIdentifier(apiOperationModel, operationIdentifierExpressionHandler),
                               buildOperationName(apiOperationModel, operationDescriptor, method,
                                                  operationDisplayNameExpressionHandler),
                               buildOperationDescription(apiOperationModel, operationDescriptor),
                               apiOperationModel.getPath(),
                               method,
                               uriParameters,
                               queryParameters,
                               headers,
                               buildInputTypeMetadata(apiOperationModel, operationDescriptor,
                                                      connectorDescriptor),
                               outputTypeMetadata,
                               securitySchemeBuilder
                                   .buildSecuritySchemes(apiOperationModel.getSecuritySchemesModel(),
                                                         connectorDescriptor),
                               buildAlternativeBaseUri(endPointDescriptor, operationDescriptor),
                               buildPagination(operationDescriptor, paginations),
                               buildSkipOutputTypeValidation(operationDescriptor),
                               buildVoidOperation(operationDescriptor),
                               buildQueryParamArrayFormat(connectorDescriptor, operationDescriptor),
                               isOperationIgnoredInDescriptor(endPointDescriptor,
                                                              operationDescriptor));

    operation.setSampleData(sampleDataBuilder.buildSampleData(getSampleDataExpressionDescriptor(operationDescriptor),
                                                              operation,
                                                              null));

    return operation;
  }

  private ResolverExpressionDescriptor<SampleDataDefinitionDescriptor> getSampleDataExpressionDescriptor(OperationDescriptor operationDescriptor) {
    return operationDescriptor != null ? operationDescriptor.getSampleDataExpressionDescriptor() : null;
  }

  /**
   * Checks if an operation is ignored in the connector descriptor. If the operation descriptor is explicitly ignored, it will
   * always be ignored. If the operation descriptor is explicitly not ignored, it never be ignored. If the operation descriptor is
   * does not define if the operation is ignored or not this will take its value from the endpoint descriptor.
   * 
   * @return true if the operation is ignored in the connector descriptor
   */
  private boolean isOperationIgnoredInDescriptor(EndPointDescriptor endPointDescriptor, OperationDescriptor operationDescriptor) {

    final boolean ignoredEndpoint =
        endPointDescriptor != null && endPointDescriptor.isIgnored() != null && endPointDescriptor.isIgnored();
    final Boolean ignoredOperation = operationDescriptor != null ? operationDescriptor.isIgnored() : null;

    return (ignoredOperation != null && ignoredOperation) || (ignoredOperation == null && ignoredEndpoint);
  }

  protected QueryParamArrayFormat buildQueryParamArrayFormat(ConnectorDescriptor connectorDescriptor,
                                                             OperationDescriptor operationDescriptor) {
    if (operationDescriptor != null && isNotBlank(operationDescriptor.getQueryParamArrayFormat())) {
      return QueryParamArrayFormat.valueOf(operationDescriptor.getQueryParamArrayFormat().toUpperCase());
    } else if (isNotBlank(connectorDescriptor.getQueryParamArrayFormat())) {
      return QueryParamArrayFormat.valueOf(connectorDescriptor.getQueryParamArrayFormat().toUpperCase());
    }
    return MULTIMAP;
  }

  private static Boolean buildSkipOutputTypeValidation(OperationDescriptor operationDescriptor) {
    return operationDescriptor != null ? operationDescriptor.getSkipOutputTypeValidation() : null;
  }

  private static Boolean buildVoidOperation(OperationDescriptor operationDescriptor) {
    return operationDescriptor != null ? operationDescriptor.getVoidOperation() : null;
  }


  private String buildOperationIdentifier(
                                          APIOperationModel apiOperationModel,
                                          OperationIdentifierExpressionHandler operationIdentifierExpressionHandler) {
    return operationIdentifierExpressionHandler.evaluate(apiOperationModel.getOperationId(),
                                                         apiOperationModel.getHttpMethod(), apiOperationModel.getPath());
  }


  private static String buildOperationName(APIOperationModel operationModel, OperationDescriptor operationDescriptor,
                                           HTTPMethod method,
                                           OperationDisplayNameExpressionHandler operationDisplayNameExpressionHandler) {
    if (operationDescriptor != null && isNotBlank(operationDescriptor.getDisplayName())) {
      return operationDescriptor.getDisplayName();
    } else {
      return operationDisplayNameExpressionHandler.evaluate(operationModel.getOperationId(), method.name(),
                                                            operationModel.getPath(), operationModel.getSummary());
    }
  }


  private static String buildOperationDescription(APIOperationModel operationModel, OperationDescriptor operationDescriptor) {
    if (operationDescriptor != null && isNotBlank(operationDescriptor.getDescription())) {
      return operationDescriptor.getDescription();
    } else {
      return operationModel.getDescription();
    }
  }

  private static String buildAlternativeBaseUri(EndPointDescriptor endPointDescriptor, OperationDescriptor operationDescriptor) {
    if (operationDescriptor != null && isNotBlank(operationDescriptor.getBaseUri())) {
      return operationDescriptor.getBaseUri();
    } else if (endPointDescriptor != null && isNotBlank(endPointDescriptor.getBaseUri())) {
      return endPointDescriptor.getBaseUri();
    }
    return null;
  }

  private static Pagination buildPagination(OperationDescriptor operationDescriptor, List<Pagination> paginations) {
    if (operationDescriptor != null && isNotBlank(operationDescriptor.getPagination())) {
      return paginations.stream().filter(x -> x.getName().equals(operationDescriptor.getPagination())).findFirst().orElse(null);
    }

    return null;
  }

  private TypeDefinition buildInputTypeMetadata(APIOperationModel apiOperationModel,
                                                OperationDescriptor operationDescriptor,
                                                ConnectorDescriptor connectorDescriptor)
      throws ModelGenerationException {

    String descriptorInputMediaType = EMPTY;
    String descriptorInputTypeSchema = EMPTY;
    List<PartDescriptor> partDescriptors = null;
    if (operationDescriptor != null) {
      descriptorInputMediaType = operationDescriptor.getInputMediaType();
      descriptorInputTypeSchema = operationDescriptor.getInputTypeSchema();
      partDescriptors = operationDescriptor.getExpects() != null ? operationDescriptor.getExpects().getPart() : null;
    }

    return buildIOTypeMetadata(apiOperationModel.getInputMetadataModel(),
                               connectorDescriptor.getDefaultInputMediaType(),
                               descriptorInputMediaType,
                               descriptorInputTypeSchema,
                               partDescriptors);
  }

  private TypeDefinition buildOutputTypeMetadata(APIOperationModel apiOperationModel,
                                                 OperationDescriptor operationDescriptor,
                                                 ConnectorDescriptor connectorDescriptor)
      throws ModelGenerationException {
    String descriptorOutputMediaType = EMPTY;
    String descriptorOutputTypeSchema = EMPTY;

    if (operationDescriptor != null) {
      descriptorOutputMediaType = operationDescriptor.getOutputMediaType();
      descriptorOutputTypeSchema = operationDescriptor.getOutputTypeSchema();
    }

    return buildIOTypeMetadata(apiOperationModel.getOutputMetadataModel(),
                               connectorDescriptor.getDefaultOutputMediaType(),
                               descriptorOutputMediaType,
                               descriptorOutputTypeSchema,
                               null /* There is no support for output multipart descriptors */);
  }

  private TypeDefinition buildIOTypeMetadata(List<APITypeModel> metadataModels,
                                             String descriptorGlobalMediaType,
                                             String descriptorMediaType,
                                             String descriptorTypeSchema,
                                             List<PartDescriptor> partDescriptors)
      throws ModelGenerationException {
    MediaType inputMediaType = null;

    if (isNotBlank(descriptorMediaType)) {
      inputMediaType = MediaType.valueOf(descriptorMediaType);
    } else if (isNotBlank(descriptorGlobalMediaType)) {
      inputMediaType = MediaType.valueOf(descriptorGlobalMediaType);
    }

    if (isNotBlank(descriptorTypeSchema)) {
      if (inputMediaType == null) {
        APITypeModel defaultApiTypeModel = getDefaultApiTypeModel(metadataModels);
        inputMediaType = defaultApiTypeModel.getMediaType();
      }

      return typeDefinitionBuilder.buildTypeDefinition(descriptorTypeSchema, inputMediaType);
    }

    return buildTypeMetadata(metadataModels, inputMediaType, partDescriptors);
  }

  private TypeDefinition buildTypeMetadata(List<APITypeModel> metadataModels, MediaType mediaType,
                                           List<PartDescriptor> partDescriptors)
      throws ModelGenerationException {

    if (!metadataModels.isEmpty()) {
      // API defines metadata media type ->
      // -- If descriptor defines one it must exist in the API spec and it will be used to generate the connector
      // -- If descriptor does not define a media type, use the default one from the API definitions
      // API does not define media type ->
      // -- Descriptor must define a media type, and it will be used for the first metadata defined in the API spec

      boolean metadataModelsDefineMediaType = metadataModels.stream().anyMatch(x -> x.getMediaType() != null);

      if (metadataModelsDefineMediaType) {
        APITypeModel apiTypeModel = getDefaultApiTypeModel(metadataModels);

        if (mediaType != null) {
          apiTypeModel = getApiTypeModelByMediaType(metadataModels, mediaType);
          if (apiTypeModel == null) {
            throw new ModelGenerationException("Could not get body with media type from api spec: " + mediaType);
          }
        }

        return typeDefinitionBuilder.buildTypeDefinition(apiTypeModel, partDescriptors);
      } else {
        APITypeModel apiTypeModel = getDefaultApiTypeModel(metadataModels);
        return typeDefinitionBuilder.buildTypeDefinition(apiTypeModel, partDescriptors, mediaType);
      }
    }

    return null;
  }

  private static APITypeModel getDefaultApiTypeModel(List<APITypeModel> metadataModels) {
    APITypeModel apiTypeModel = getApiTypeModelByMediaType(metadataModels, MediaType.APPLICATION_JSON_TYPE);

    if (apiTypeModel == null) {
      apiTypeModel = getApiTypeModelByMediaType(metadataModels, MediaType.APPLICATION_XML_TYPE);
    }

    return apiTypeModel == null ? metadataModels.get(0) : apiTypeModel;
  }

  private static APITypeModel getApiTypeModelByMediaType(List<APITypeModel> metadataModels, MediaType mediaType) {
    return metadataModels.stream()
        .filter(x -> x.getMediaType() != null && x.getMediaType().equals(mediaType))
        .findFirst().orElse(null);
  }
}
