/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.util.NamingUtil.isFriendlyName;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.resolver.valueprovider.ConnectorValueProviderBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.Parameter;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.PartParameter;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.resolver.ResolverExpression;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.TypeDefinition;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.util.NamingUtil;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.valueprovider.ValueProviderDefinition;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ParameterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.PartDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.SecuritySchemeDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.exception.ModelGenerationException;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIParameterModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.type.APIParameterType;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.type.APITypeModel;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;

import javax.annotation.Nullable;
import javax.ws.rs.core.MediaType;

public class ConnectorParameterBuilder {

  private final ConnectorTypeDefinitionBuilder typeDefinitionBuilder;
  private final ConnectorValueProviderBuilder connectorValueProviderBuilder = new ConnectorValueProviderBuilder();

  public ConnectorParameterBuilder(ConnectorTypeDefinitionBuilder connectorTypeDefinitionBuilder) {
    this.typeDefinitionBuilder = connectorTypeDefinitionBuilder;
  }

  public List<Parameter> buildParameterList(List<APIParameterModel> parametersModel,
                                            OperationDescriptor operationDescriptor)
      throws ModelGenerationException {

    List<Parameter> parameterList = new ArrayList<>();
    List<String> parameterCollisionInternalNames = new ArrayList<>();
    for (APIParameterModel parameterModel : parametersModel) {
      ParameterDescriptor parameterDescriptor = getParameterDescriptor(parameterModel, operationDescriptor);
      if (parameterDescriptor == null || !parameterDescriptor.isIgnored()) {
        Parameter parameter = buildParameter(parameterModel, parameterDescriptor, parameterCollisionInternalNames);
        parameterList.add(parameter);
        parameterCollisionInternalNames.add(parameter.getInternalName());
      }
    }

    return parameterList;
  }

  public List<Parameter> buildParameterList(List<APIParameterModel> parametersModel,
                                            SecuritySchemeDescriptor securitySchemeDescriptor)
      throws ModelGenerationException {

    List<Parameter> parameterList = new ArrayList<>();
    List<String> parameterCollisionInternalNames = new ArrayList<>();
    for (APIParameterModel parameterModel : parametersModel) {
      ParameterDescriptor parameterDescriptor = getParameterDescriptor(parameterModel, securitySchemeDescriptor);
      if (parameterDescriptor == null || !parameterDescriptor.isIgnored()) {
        Parameter parameter = buildParameter(parameterModel, parameterDescriptor, parameterCollisionInternalNames);
        parameterList.add(parameter);
        parameterCollisionInternalNames.add(parameter.getInternalName());
      }
    }

    return parameterList;
  }

  public List<Parameter> buildParameterList(List<ParameterDescriptor> parameterDescriptors,
                                            APIParameterType parameterType) {

    List<Parameter> parameterList = new ArrayList<>();
    List<String> parameterCollisionInternalNames = new ArrayList<>();
    for (ParameterDescriptor parameterDescriptor : parameterDescriptors) {
      if (parameterDescriptor.isAdditional()) {
        Parameter parameter = buildParameter(parameterDescriptor, parameterType, parameterCollisionInternalNames);
        parameterList.add(parameter);
        parameterCollisionInternalNames.add(parameter.getInternalName());
      }
    }
    return parameterList;
  }

  private static ParameterDescriptor getParameterDescriptor(APIParameterModel parameterModel,
                                                            SecuritySchemeDescriptor securitySchemeDescriptor) {
    if (securitySchemeDescriptor == null) {
      return null;
    }

    List<ParameterDescriptor> parameterDescriptors = new LinkedList<>();

    if (securitySchemeDescriptor.getHeaders() != null) {
      if (parameterModel.getParameterType().equals(APIParameterType.HEADER)) {
        parameterDescriptors = securitySchemeDescriptor.getHeaders();
      }
      if (parameterModel.getParameterType().equals(APIParameterType.QUERY)) {
        parameterDescriptors = securitySchemeDescriptor.getQueryParameters();
      }
    }

    return parameterDescriptors.stream()
        .filter(x -> isNotBlank(x.getParamName()))
        .filter(x -> x.getParamName().equalsIgnoreCase(parameterModel.getExternalName()))
        .findFirst()
        .orElse(null);
  }

  private static ParameterDescriptor getParameterDescriptor(APIParameterModel parameterModel,
                                                            OperationDescriptor operationDescriptor) {
    if (operationDescriptor == null) {
      return null;
    }

    List<ParameterDescriptor> parameterDescriptors = new LinkedList<>();

    if (operationDescriptor.getExpects() != null) {
      if (parameterModel.getParameterType().equals(APIParameterType.URI)) {
        parameterDescriptors = operationDescriptor.getExpects().getUriParameter();
      } else if (parameterModel.getParameterType().equals(APIParameterType.QUERY)) {
        parameterDescriptors = operationDescriptor.getExpects().getQueryParameter();
      } else if (parameterModel.getParameterType().equals(APIParameterType.HEADER)) {
        parameterDescriptors = operationDescriptor.getExpects().getHeader();
      }
    }

    return parameterDescriptors.stream()
        .filter(x -> isNotBlank(x.getParamName()))
        .filter(x -> x.getParamName().equalsIgnoreCase(parameterModel.getExternalName()))
        .findFirst()
        .orElse(null);
  }

  public Parameter buildParameter(APIParameterModel apiParameterModel, ParameterDescriptor parameterDescriptor,
                                  List<String> collisionInternalNames)
      throws ModelGenerationException {

    return new Parameter(buildDisplayName(apiParameterModel, parameterDescriptor),
                         apiParameterModel.getExternalName(),
                         buildParameterType(apiParameterModel.getParameterType()),
                         buildParameterTypeDefinition(apiParameterModel.getTypeModel(), null),
                         buildDescription(apiParameterModel, parameterDescriptor),
                         apiParameterModel.isRequired(),
                         apiParameterModel.getDefaultValue(),
                         apiParameterModel.isPassword(),
                         buildValueProvider(parameterDescriptor),
                         collisionInternalNames);
  }

  private Parameter buildParameter(ParameterDescriptor parameterDescriptor, APIParameterType parameterType,
                                   List<String> collisionNames) {
    return new Parameter(parameterDescriptor.getDisplayName(),
                         parameterDescriptor.getParamName(),
                         buildParameterType(parameterType),
                         typeDefinitionBuilder.buildPrimitiveType(parameterDescriptor.getDataType()),
                         parameterDescriptor.getDescription(),
                         parameterDescriptor.getRequired(),
                         parameterDescriptor.getDefaultValue(),
                         false,
                         buildValueProvider(parameterDescriptor),
                         collisionNames);
  }


  private ResolverExpression<ValueProviderDefinition> buildValueProvider(ParameterDescriptor parameterDescriptor) {
    if (parameterDescriptor == null) {
      return null;
    }

    return connectorValueProviderBuilder.buildValueProvider(parameterDescriptor.getValueProvider());
  }

  private ParameterType buildParameterType(APIParameterType parameterType) {
    switch (parameterType) {
      case URI:
        return ParameterType.URI;
      case QUERY:
        return ParameterType.QUERY;
      case HEADER:
        return ParameterType.HEADER;
      case SECURITY:
        return ParameterType.SECURITY;
      case PART:
        return ParameterType.PART;
      default:
        throw new IllegalArgumentException("Parameter type not supported. Should not reach here.");
    }
  }

  private static String buildDisplayName(APIParameterModel apiParameterModel, ParameterDescriptor parameterDescriptor) {
    return buildDisplayName(apiParameterModel, parameterDescriptor != null ? parameterDescriptor.getDisplayName() : EMPTY);
  }

  private static String buildDisplayName(APIParameterModel apiParameterModel, PartDescriptor partDescriptor) {
    return buildDisplayName(apiParameterModel, partDescriptor != null ? partDescriptor.getDisplayName() : EMPTY);
  }

  private static String buildDisplayName(APIParameterModel apiParameterModel, String descriptorValue) {
    String displayName;

    if (isNotBlank(descriptorValue)) {
      displayName = descriptorValue;
    } else if (isNotBlank(apiParameterModel.getDisplayName())) {
      displayName = apiParameterModel.getDisplayName();
    } else {
      displayName = apiParameterModel.getExternalName();
      if (!isFriendlyName(displayName)) {
        // TODO RSDK-368: Display name generation by comprehension
        displayName = NamingUtil.makeNameFriendly(displayName);
      }
    }
    return displayName;

  }

  private static String buildDescription(APIParameterModel apiParameterModel, ParameterDescriptor parameterDescriptor) {
    if (parameterDescriptor != null && isNotBlank(parameterDescriptor.getDescription())) {
      return parameterDescriptor.getDescription();
    } else {
      return apiParameterModel.getDescription();
    }
  }

  private TypeDefinition buildParameterTypeDefinition(APITypeModel typeModel, @Nullable MediaType mediaType)
      throws ModelGenerationException {

    if (mediaType == null) {
      mediaType = typeModel.getMediaType() != null ? typeModel.getMediaType() : MediaType.TEXT_PLAIN_TYPE;
    }

    return typeDefinitionBuilder.buildTypeDefinition(typeModel, null, mediaType);
  }

  public PartParameter buildPartParameter(APIParameterModel apiParameterModel, @Nullable PartDescriptor partDescriptor)
      throws ModelGenerationException {
    TypeDefinition partTypeDefinition;

    MediaType mediaType = null;
    if (partDescriptor != null && isNotBlank(partDescriptor.getContentType())) {
      mediaType = MediaType.valueOf(partDescriptor.getContentType());
    }

    if (partDescriptor != null && isNotBlank(partDescriptor.getInputType())) {
      partTypeDefinition = typeDefinitionBuilder.buildTypeDefinition(partDescriptor.getInputType(), mediaType);
    } else {
      partTypeDefinition = buildParameterTypeDefinition(apiParameterModel.getTypeModel(), mediaType);
    }

    return new PartParameter(
                             buildDisplayName(apiParameterModel, partDescriptor),
                             apiParameterModel.getExternalName(),
                             partTypeDefinition,
                             (partDescriptor != null && isNotBlank(partDescriptor.getDescription()))
                                 ? partDescriptor.getDescription()
                                 : apiParameterModel.getDescription(),
                             partDescriptor != null && partDescriptor.isFilePart());
  }
}
