/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.resolver.sampledata;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.HTTPMethod.fromString;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.HEADER;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.QUERY;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.URI;
import static java.lang.String.format;
import static java.util.stream.Collectors.toList;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorOperation;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.resolver.ConnectorDataExpressionBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.resolver.ConnectorResolverBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dataexpression.httprequest.HttpRequestBinding;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dataexpression.httprequest.HttpRequestDataExpression;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dataexpression.script.DwScriptDataExpression;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.expression.DataWeaveExpression;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.generic.Argument;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.Parameter;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.resolver.ResolverExpression;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.sampledata.SampleDataDefinition;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.trigger.Trigger;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.resolvers.ResolverDefinitionDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.resolvers.ResolverExpressionDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.resolvers.ResolverReferenceDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.sampledata.SampleDataDefinitionDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.sampledata.SampleDataSameOperationDefinitionDescriptor;

import java.util.List;

public class ConnectorSampleDataBuilder {

  private final ConnectorResolverBuilder connectorResolverBuilder = new ConnectorResolverBuilder();
  private final ConnectorDataExpressionBuilder connectorDataExpressionBuilder = new ConnectorDataExpressionBuilder();

  public ResolverExpression<SampleDataDefinition> buildSampleData(ResolverExpressionDescriptor<SampleDataDefinitionDescriptor> sampleDataExpressionDescriptor,
                                                                  ConnectorOperation connectorOperation,
                                                                  Trigger trigger) {
    if (sampleDataExpressionDescriptor == null) {
      return null;
    }

    if (sampleDataExpressionDescriptor instanceof ResolverReferenceDescriptor<?>) {
      ResolverReferenceDescriptor<SampleDataDefinitionDescriptor> resolverReference =
          (ResolverReferenceDescriptor<SampleDataDefinitionDescriptor>) sampleDataExpressionDescriptor;

      return connectorResolverBuilder
          .buildResolverReference(resolverReference,
                                  buildSampleDataDefinition(resolverReference.getDeclaration().getDefinition()));

    } else if (sampleDataExpressionDescriptor instanceof ResolverDefinitionDescriptor<?>) {
      SampleDataDefinitionDescriptor definition = (SampleDataDefinitionDescriptor) sampleDataExpressionDescriptor;

      if (definition instanceof SampleDataSameOperationDefinitionDescriptor) {
        if (connectorOperation != null) {
          return buildSampleDataSameOperationDefinition(connectorOperation);
        } else if (trigger != null) {
          return buildSampleDataSameOperationDefinition(trigger);
        } else {
          throw new IllegalArgumentException("SampleData target not supported. This is a bug.");
        }
      } else {
        return buildSampleDataDefinition(definition);
      }
    }

    throw new IllegalArgumentException("ResolverExpressionDescriptor not supported, this is a bug.");
  }

  private SampleDataDefinition buildSampleDataSameOperationDefinition(ConnectorOperation connectorOperation) {
    DwScriptDataExpression paginationExpression = null;
    if (connectorOperation.hasPagination()) {
      paginationExpression = new DwScriptDataExpression(connectorOperation.getPagination().getPaginationResponseExpression());
    }

    return new SampleDataDefinition(new HttpRequestDataExpression(connectorOperation.getPath(),
                                                                  fromString(connectorOperation.getHttpMethod()),
                                                                  buildBindings(connectorOperation)),
                                    paginationExpression);
  }

  private SampleDataDefinition buildSampleDataSameOperationDefinition(Trigger trigger) {
    return new SampleDataDefinition(new HttpRequestDataExpression(trigger.getOperation().getPath(),
                                                                  fromString(trigger.getOperation().getHttpMethod()),
                                                                  buildTriggerBindings(trigger)),
                                    new DwScriptDataExpression(getExpressionForFirstItem(trigger.getItemsExpression())));
  }

  /**
   * Modifies an expression that returns an array and makes it return it's first element.
   *
   * i.e:
   *
   * #[payload.items] -> (payload.items)[0].
   *
   * #[payload.data default []] -> (payload.data default [])[0].
   *
   * payload.data default [] -> (payload.data default [])[0].
   *
   */
  private String getExpressionForFirstItem(String expression) {
    if (expression.startsWith("#[") && expression.endsWith("]")) {
      expression = expression.substring(2, expression.length() - 1);
    }

    return "(" + expression + ")[0]";
  }

  private HttpRequestBinding buildTriggerBindings(Trigger trigger) {
    if (trigger.getParameterBindings() == null || trigger.getParameterBindings().isEmpty()) {
      return null;
    }

    return new HttpRequestBinding(buildTriggerArguments(trigger, URI),
                                  buildTriggerArguments(trigger, QUERY),
                                  buildTriggerArguments(trigger, HEADER));
  }

  private List<Argument> buildTriggerArguments(Trigger trigger, ParameterType parameterType) {
    return trigger.getParameterBindings().stream()
        .filter(x -> x.getParameterType().equals(parameterType))
        .map(x -> buildArgument(x.getName(), x.getExpression()))
        .collect(toList());
  }

  private HttpRequestBinding buildBindings(ConnectorOperation connectorOperation) {
    List<Parameter> requiredUriParams = getRequiredParameters(connectorOperation.getUriParameters());
    List<Parameter> requiredQueryParams = getRequiredParameters(connectorOperation.getQueryParameters());
    List<Parameter> requiredHeaders = getRequiredParameters(connectorOperation.getHeaders());

    if (requiredUriParams.isEmpty() && requiredQueryParams.isEmpty() && requiredHeaders.isEmpty()) {
      return null;
    }

    return new HttpRequestBinding(buildArguments(requiredUriParams, URI),
                                  buildArguments(requiredQueryParams, QUERY),
                                  buildArguments(requiredHeaders, HEADER));
  }

  private List<Parameter> getRequiredParameters(List<Parameter> parameterList) {
    return parameterList.stream().filter(Parameter::isRequired).collect(toList());
  }

  private List<Argument> buildArguments(List<Parameter> parameters, ParameterType parameterType) {
    return parameters.stream().map(x -> buildArgument(x.getExternalName(), parameterType)).collect(toList());
  }

  private Argument buildArgument(String externalName, ParameterType parameterType) {
    return buildArgument(externalName, format("%s.%s", parameterType.getAccessorName(), externalName));
  }

  private Argument buildArgument(String externalName, String expression) {
    return new Argument(externalName, new DataWeaveExpression(expression));
  }

  private SampleDataDefinition buildSampleDataDefinition(SampleDataDefinitionDescriptor sampleDataDefinitionDescriptor) {
    return new SampleDataDefinition(
                                    connectorDataExpressionBuilder
                                        .buildHttpRequestDataExpression(sampleDataDefinitionDescriptor.getResult()),
                                    connectorDataExpressionBuilder
                                        .buildScriptDataExpression(sampleDataDefinitionDescriptor.getTransform()));
  }
}
