/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.ConnectorSecurityScheme.SecuritySchemeType.UNSECURED;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.Parameter;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.util.HashCodeUtil;

import java.util.LinkedList;
import java.util.List;

public abstract class ConnectorSecurityScheme {

  public enum SecuritySchemeType {

    BASIC("Basic Authentication"), OAUTH1("OAuth 1.0"), OAUTH2("OAuth 2.0"), PASS_THROUGH("Pass Through"), DIGEST_AUTHENTICATION(
        "Digest Authentication"), CUSTOM_AUTHENTICATION(
            "Custom Authentication"), BEARER("Bearer Authentication"), UNSECURED("Unsecured"), NOT_SUPPORTED("Not Supported");

    private final String type;

    SecuritySchemeType(String type) {
      this.type = type;
    }

    @Override
    public String toString() {
      return this.type;
    }
  }

  protected final List<Parameter> headers = new LinkedList<>();
  protected final List<Parameter> queryParameters = new LinkedList<>();

  protected final SecuritySchemeType schemeType;
  protected final String displayName;
  protected final TestConnectionConfig testConnectionConfig;

  public ConnectorSecurityScheme(SecuritySchemeType schemeType, String displayName, TestConnectionConfig testConnectionConfig) {
    this.schemeType = schemeType;
    this.displayName = displayName;
    this.testConnectionConfig = testConnectionConfig;
  }

  public SecuritySchemeType getSchemeType() {
    return this.schemeType;
  }

  public String getDisplayName() {
    if (schemeType.equals(UNSECURED)) {
      return UNSECURED.name();
    } else {
      return displayName;
    }
  }

  public TestConnectionConfig getTestConnectionConfig() {
    return testConnectionConfig;
  }

  public List<Parameter> getQueryParameters() {
    return queryParameters;
  }

  public List<Parameter> getHeaders() {
    return headers;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o)
      return true;
    if (!(o instanceof ConnectorSecurityScheme))
      return false;

    ConnectorSecurityScheme that = (ConnectorSecurityScheme) o;

    return that.hashCode() == this.hashCode();
  }

  @Override
  public int hashCode() {
    return HashCodeUtil.generateHashCode(schemeType, headers, queryParameters);
  }

  public static Builder builder(String displayName, List<Parameter> customQueryParams, List<Parameter> customHeaders,
                                TestConnectionConfig testConnectionConfig) {
    return new Builder(displayName, customQueryParams, customHeaders, testConnectionConfig);
  }

  public static class Builder {

    private final List<Parameter> customHeaders;
    private final List<Parameter> customQueryParams;
    private final String displayName;
    private final TestConnectionConfig testConnectionConfig;

    public Builder(String displayName, List<Parameter> customQueryParams, List<Parameter> customHeaders,
                   TestConnectionConfig testConnectionConfig) {
      this.customHeaders = customHeaders;
      this.customQueryParams = customQueryParams;
      this.displayName = displayName;
      this.testConnectionConfig = testConnectionConfig;
    }

    public BasicAuthScheme buildBasicAuthScheme() {
      return new BasicAuthScheme(displayName, testConnectionConfig);
    }

    public CustomAuthenticationScheme buildCustomAuthenticationScheme() {
      return new CustomAuthenticationScheme(displayName, customQueryParams, customHeaders, testConnectionConfig);
    }

    public DigestAuthenticationScheme buildDigestAuthenticationSchemeScheme() {
      return new DigestAuthenticationScheme(displayName, testConnectionConfig);
    }

    public OAuth2AuthorizationCodeScheme buildOAuth2AuthorizationCodeScheme(String authorizationUri, String accessTokenUri,
                                                                            List<String> scopes,
                                                                            String refreshTokenConditionExpression) {
      return new OAuth2AuthorizationCodeScheme(displayName,
                                               authorizationUri,
                                               accessTokenUri,
                                               scopes,
                                               customQueryParams,
                                               customHeaders,
                                               testConnectionConfig, refreshTokenConditionExpression);
    }

    public OAuth2ClientCredentialsScheme buildOAuth2ClientCredentialsScheme(String authorizationUri, String accessTokenUri,
                                                                            List<String> scopes,
                                                                            String refreshTokenConditionExpression) {
      return new OAuth2ClientCredentialsScheme(displayName,
                                               authorizationUri,
                                               accessTokenUri,
                                               scopes,
                                               customQueryParams,
                                               customHeaders,
                                               testConnectionConfig, refreshTokenConditionExpression);
    }

    public PassThroughScheme buildPassThroughScheme() {
      return new PassThroughScheme(displayName, customQueryParams, customHeaders, testConnectionConfig);
    }

    public UnsecuredScheme buildUnsecuredScheme() {
      return new UnsecuredScheme(testConnectionConfig);
    }

    public BearerAuthScheme buildBearerAuthScheme() {
      return new BearerAuthScheme(displayName, testConnectionConfig);
    }
  }
}
