/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.PartParameter;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.exception.ModelGenerationException;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.schema.TypeSchema;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.util.HashCodeUtil;

import javax.ws.rs.core.MediaType;
import java.util.List;

public abstract class TypeDefinition {

  protected final MediaType mediaType;
  protected final String example;
  protected final List<String> enumValues;
  protected final TypeSchema typeSchema;

  protected final String displayName;
  protected final String description;

  TypeDefinition(MediaType mediaType, String example, List<String> enumValues, TypeSchema typeSchema, String displayName,
                 String description) {
    this.mediaType = mediaType;
    this.example = example;
    this.enumValues = enumValues;
    this.typeSchema = typeSchema;

    this.displayName = displayName;
    this.description = description;
  }

  public MediaType getMediaType() {
    return mediaType;
  }

  public String getExample() {
    return example;
  }

  public List<String> getEnumValues() {
    return enumValues;
  }

  public TypeSchema getTypeSchema() {
    return typeSchema;
  }

  public boolean isEnum() {
    return enumValues != null && !enumValues.isEmpty();
  }

  public String getDisplayName() {
    return displayName;
  }

  public String getDescription() {
    return description;
  }

  public static TypeDefinition simplePrimitiveType(PrimitiveTypeDefinition.PrimitiveType type) {
    return new PrimitiveTypeDefinition(MediaType.TEXT_PLAIN_TYPE, null, null, null, type, null, null);
  }

  public static TypeDefinition simpleStringType() {
    return simplePrimitiveType(PrimitiveTypeDefinition.PrimitiveType.STRING);
  }

  @Override
  public int hashCode() {
    String mediaTypeString = this.mediaType != null ? this.mediaType.toString() : null;
    return HashCodeUtil.generateHashCode(mediaTypeString, example, enumValues, typeSchema);
  }

  public static Builder builder(MediaType mediaType, TypeSchema typeSchema, List<String> enumValues, String example)
      throws ModelGenerationException {
    if (mediaType == null) {
      throw new ModelGenerationException("Media Type can not be empty");
    }
    return new Builder(mediaType, typeSchema, enumValues, example);
  }

  public static class Builder {

    private final MediaType mediaType;
    private final TypeSchema typeSchema;
    private final List<String> enumValues;
    private final String example;
    private String displayName;
    private String description;

    Builder(MediaType mediaType, TypeSchema typeSchema, List<String> enumValues, String example) {
      this.mediaType = mediaType;
      this.typeSchema = typeSchema;
      this.enumValues = enumValues;
      this.example = example;
    }

    public ArrayTypeDefinition buildArrayType(TypeDefinition innerType) {
      return new ArrayTypeDefinition(mediaType, example, enumValues, typeSchema, innerType, displayName, description);
    }

    public MultipartTypeDefinition buildMultipartTypeDefinition(List<PartParameter> parts) {
      return new MultipartTypeDefinition(example, enumValues, typeSchema, parts, displayName, description);
    }

    public ObjectTypeDefinition buildObjectTypeDefinition() {
      return new ObjectTypeDefinition(mediaType, example, enumValues, typeSchema, displayName, description);
    }

    public PrimitiveTypeDefinition buildPrimitiveTypeDefinition(PrimitiveTypeDefinition.PrimitiveType primitiveType) {
      return new PrimitiveTypeDefinition(mediaType, example, enumValues, typeSchema, primitiveType, displayName, description);
    }

    public UnionTypeDefinition buildUnionTypeDefinition(List<TypeDefinition> unionTypes) {
      return new UnionTypeDefinition(mediaType, example, enumValues, typeSchema, unionTypes, displayName, description);
    }

    public EmptyTypeDefinition buildEmptyTypeDefinition() {
      return new EmptyTypeDefinition(mediaType, example, enumValues, displayName, description);
    }

    public void withDisplayName(String displayName) {
      this.displayName = displayName;
    }

    public void withDescription(String description) {
      this.description = description;
    }
  }
}
