/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.OAuth2Scheme.GrantType.AUTHORIZATION_CODE;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.OAuth2Scheme.GrantType.CLIENT_CREDENTIALS;
import static java.util.Objects.requireNonNull;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.ObjectUtils.defaultIfNull;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.ConnectorSecurityScheme.SecuritySchemeType.CUSTOM_AUTHENTICATION;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.ConnectorSecurityScheme.SecuritySchemeType.UNSECURED;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.TypeSchemaPool;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dw.ParameterIdentifierExpressionHandler;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.BasicAuthDefaultParameters;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.BearerAuthDefaultParameters;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.DigestAuthDefaultParameters;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.Parameter;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.SecurityDefaultParameters;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.BasicAuthScheme;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.BearerAuthScheme;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.ConnectorSecurityScheme;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.CustomAuthenticationScheme;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.MockedAuthenticationScheme;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.DigestAuthenticationScheme;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.OAuth2AuthorizationCodeScheme;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.OAuth2ClientCredentialsScheme;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.OAuth2Scheme;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.PassThroughScheme;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.TestConnectionConfig;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.UnsecuredScheme;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.util.ComparisonUtil;


import java.util.ArrayList;
import java.util.List;

public class SecuritySchemeBuilder {

  private ConnectorSecurityScheme.SecuritySchemeType type;
  private final String name;

  private final List<ParameterBuilder> nonStandardHeaderBuilders = new ArrayList<>();
  private final List<ParameterBuilder> nonStandardQueryParameterBuilders = new ArrayList<>();

  private final SecurityDefaultParametersBuilder securityDefaultParametersBuilder = new SecurityDefaultParametersBuilder();

  private String alias;
  private TestConnectionConfig testConnectionConfig;
  private String authorizationUri;
  private String accessTokenUri;
  private List<String> scopes;
  private String refreshTokenConditionExpression;
  private OAuth2Scheme.GrantType grantType;
  private Boolean ignored;
  private Boolean refined;

  private String fqn;

  public SecuritySchemeBuilder(String name, ConnectorSecurityScheme.SecuritySchemeType type) {
    requireNonNull(name);
    requireNonNull(type);

    this.type = type;
    this.name = name;
  }

  public String getName() {
    return name;
  }

  public String getAlias() {
    return alias;
  }

  public ConnectorSecurityScheme.SecuritySchemeType getType() {
    return type;
  }

  public Boolean isIgnored() {
    return ignored;
  }

  public ParameterBuilder getOrCreateParameterBuilder(ParameterType parameterType, String name) {
    // TODO code smell, quite similar to
    // com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.OperationBuilder.getOrCreateParameterBuilder
    List<ParameterBuilder> parameterBuilders = getParameterBuilders(parameterType);

    ParameterBuilder paramBuilder = parameterBuilders.stream()
        .filter(x -> ComparisonUtil.externalNameParamsComparison(x.getExternalName(), name, parameterType))
        .findFirst().orElse(null);

    if (paramBuilder == null) {
      String parameterIdentifier = null; // No parameter identifier for a schema type of parameter
      paramBuilder = new ParameterBuilder(parameterType, name, parameterIdentifier);
      parameterBuilders.add(paramBuilder);
    }

    return paramBuilder;
  }

  private List<ParameterBuilder> getParameterBuilders(ParameterType parameterType) {
    switch (parameterType) {
      case QUERY:
        return nonStandardQueryParameterBuilders;
      case HEADER:
        return nonStandardHeaderBuilders;
      default:
        throw new IllegalArgumentException("Parameter Type not supported. This is a bug.");
    }
  }


  public SecuritySchemeBuilder testConnectionConfig(TestConnectionConfig testConnectionConfig) {
    this.testConnectionConfig = defaultIfNull(testConnectionConfig, this.testConnectionConfig);
    return this;
  }

  public SecurityDefaultParametersBuilder getSecurityDefaultParametersBuilder() {
    return securityDefaultParametersBuilder;
  }

  public SecuritySchemeBuilder authorizationUri(String authorizationUri) {
    this.authorizationUri = defaultIfNull(authorizationUri, this.authorizationUri);
    return this;
  }

  public SecuritySchemeBuilder accessTokenUri(String accessTokenUri) {
    this.accessTokenUri = defaultIfNull(accessTokenUri, this.accessTokenUri);
    return this;
  }

  public SecuritySchemeBuilder scopes(List<String> scopes) {
    this.scopes = defaultIfNull(scopes, this.scopes);
    return this;
  }

  public SecuritySchemeBuilder refreshTokenConditionExpression(String refreshTokenConditionExpression) {
    this.refreshTokenConditionExpression = defaultIfNull(refreshTokenConditionExpression, this.refreshTokenConditionExpression);
    return this;
  }

  public SecuritySchemeBuilder grantType(OAuth2Scheme.GrantType grantType) {
    this.grantType = defaultIfNull(grantType, this.grantType);
    return this;
  }

  public SecuritySchemeBuilder ignored(Boolean ignored) {
    this.ignored = defaultIfNull(ignored, this.ignored);
    return this;
  }

  public SecuritySchemeBuilder refined(Boolean refined) {
    this.refined = defaultIfNull(refined, this.refined);
    return this;
  }

  public SecuritySchemeBuilder alias(String alias) {
    this.alias = alias;
    return this;
  }

  public SecuritySchemeBuilder fqn(String fqn) {
    this.fqn = fqn;
    return this;
  }

  protected boolean isRefined() {
    return defaultIfNull(refined, false);
  }

  public ConnectorSecurityScheme build(TypeSchemaPool typeSchemaPool, TestConnectionConfig defaultTestConnectionConfig,
                                       ParameterIdentifierExpressionHandler parameterIdExpressionHandler) {
    TestConnectionConfig testConnectionConfig = defaultIfNull(this.testConnectionConfig, defaultTestConnectionConfig);

    switch (type) {
      case BASIC:
        return buildBasicAuthScheme(testConnectionConfig);
      case CUSTOM_AUTHENTICATION:
        return buildCustomAuthenticationScheme(typeSchemaPool, parameterIdExpressionHandler, testConnectionConfig);
      case DIGEST_AUTHENTICATION:
        return buildDigestAuthenticationSchemeScheme(testConnectionConfig);
      case PASS_THROUGH:
        return buildPassThroughScheme(typeSchemaPool, parameterIdExpressionHandler,
                                      testConnectionConfig);
      case OAUTH2:
        return buildOauth2Scheme(typeSchemaPool, parameterIdExpressionHandler,
                                 testConnectionConfig);
      case BEARER:
        return buildBearerAuthScheme(testConnectionConfig);
      case MOCKED:
        return buildMockedScheme();
      case UNSECURED:
      case NOT_SUPPORTED:
        return buildUnsecuredScheme(testConnectionConfig);
      default:
        throw new IllegalArgumentException("Security scheme '" + type + "' is not supported. This is a bug.");
    }
  }

  private ConnectorSecurityScheme buildOauth2Scheme(TypeSchemaPool typeSchemaPool,
                                                    ParameterIdentifierExpressionHandler parameterIdExpressionHandler,
                                                    TestConnectionConfig testConnectionConfig) {
    requireNonNull(grantType);

    if (grantType.equals(AUTHORIZATION_CODE)) {
      return buildOAuth2AuthorizationCodeScheme(typeSchemaPool, parameterIdExpressionHandler,
                                                testConnectionConfig);
    } else if (grantType.equals(CLIENT_CREDENTIALS)) {
      return buildOAuth2ClientCredentialsScheme(typeSchemaPool, parameterIdExpressionHandler,
                                                testConnectionConfig);
    }

    throw new IllegalArgumentException("Grant type not supported. This is a bug.");
  }

  private BasicAuthScheme buildBasicAuthScheme(TestConnectionConfig testConnectionConfig) {
    SecurityDefaultParameters securityDefaultParameters = securityDefaultParametersBuilder.build();
    if (securityDefaultParameters == null || securityDefaultParameters instanceof BasicAuthDefaultParameters) {
      return new BasicAuthScheme(name,
                                 alias,
                                 (BasicAuthDefaultParameters) securityDefaultParameters,
                                 testConnectionConfig,
                                 isRefined());
    }
    throw new IllegalArgumentException("Default parameters not corresponding to the security scheme type: basic. This is a bug.");
  }

  private CustomAuthenticationScheme buildCustomAuthenticationScheme(TypeSchemaPool typeSchemaPool,
                                                                     ParameterIdentifierExpressionHandler parameterIdExpressionHandler,
                                                                     TestConnectionConfig testConnectionConfig) {
    return new CustomAuthenticationScheme(name,
                                          alias,
                                          buildNonStandardQueryParameters(typeSchemaPool, parameterIdExpressionHandler),
                                          buildNonStandardHeaders(typeSchemaPool, parameterIdExpressionHandler),
                                          testConnectionConfig,
                                          isRefined(),
                                          fqn);
  }

  private MockedAuthenticationScheme buildMockedScheme() {
    return new MockedAuthenticationScheme(name);
  }

  private DigestAuthenticationScheme buildDigestAuthenticationSchemeScheme(TestConnectionConfig testConnectionConfig) {
    SecurityDefaultParameters securityDefaultParameters = securityDefaultParametersBuilder.build();
    if (securityDefaultParameters == null || securityDefaultParameters instanceof DigestAuthDefaultParameters) {
      return new DigestAuthenticationScheme(name,
                                            alias,
                                            (DigestAuthDefaultParameters) securityDefaultParameters,
                                            testConnectionConfig,
                                            isRefined());
    }
    throw new IllegalArgumentException("Default parameters not corresponding to the security scheme type: digest. This is a bug.");
  }

  private OAuth2AuthorizationCodeScheme buildOAuth2AuthorizationCodeScheme(TypeSchemaPool typeSchemaPool,
                                                                           ParameterIdentifierExpressionHandler parameterIdExpressionHandler,
                                                                           TestConnectionConfig testConnectionConfig) {
    return new OAuth2AuthorizationCodeScheme(name,
                                             alias,
                                             authorizationUri,
                                             accessTokenUri,
                                             scopes,
                                             buildNonStandardQueryParameters(typeSchemaPool, parameterIdExpressionHandler),
                                             buildNonStandardHeaders(typeSchemaPool, parameterIdExpressionHandler),
                                             testConnectionConfig,
                                             refreshTokenConditionExpression,
                                             isRefined());
  }

  private OAuth2ClientCredentialsScheme buildOAuth2ClientCredentialsScheme(TypeSchemaPool typeSchemaPool,
                                                                           ParameterIdentifierExpressionHandler parameterIdExpressionHandler,
                                                                           TestConnectionConfig testConnectionConfig) {
    return new OAuth2ClientCredentialsScheme(name,
                                             alias,
                                             authorizationUri,
                                             accessTokenUri,
                                             scopes,
                                             buildNonStandardQueryParameters(typeSchemaPool, parameterIdExpressionHandler),
                                             buildNonStandardHeaders(typeSchemaPool, parameterIdExpressionHandler),
                                             testConnectionConfig,
                                             refreshTokenConditionExpression,
                                             isRefined());
  }

  private PassThroughScheme buildPassThroughScheme(TypeSchemaPool typeSchemaPool,
                                                   ParameterIdentifierExpressionHandler parameterIdExpressionHandler,
                                                   TestConnectionConfig testConnectionConfig) {
    return new PassThroughScheme(name,
                                 alias,
                                 buildNonStandardQueryParameters(typeSchemaPool, parameterIdExpressionHandler),
                                 buildNonStandardHeaders(typeSchemaPool, parameterIdExpressionHandler),
                                 testConnectionConfig,
                                 isRefined());
  }

  private UnsecuredScheme buildUnsecuredScheme(TestConnectionConfig testConnectionConfig) {
    return new UnsecuredScheme(testConnectionConfig, isRefined());
  }

  private BearerAuthScheme buildBearerAuthScheme(TestConnectionConfig testConnectionConfig) {
    SecurityDefaultParameters securityDefaultParameters = securityDefaultParametersBuilder.build();
    if (securityDefaultParameters == null || securityDefaultParameters instanceof BearerAuthDefaultParameters) {
      return new BearerAuthScheme(name,
                                  alias,
                                  (BearerAuthDefaultParameters) securityDefaultParameters,
                                  testConnectionConfig,
                                  isRefined());
    }
    throw new IllegalArgumentException("Default parameters not corresponding to the security scheme type: bearer. This is a bug.");
  }

  private List<Parameter> buildNonStandardQueryParameters(TypeSchemaPool typeSchemaPool,
                                                          ParameterIdentifierExpressionHandler parameterIdExpressionHandler) {
    return nonStandardQueryParameterBuilders.stream()
        .filter(x -> x.isIgnored() == null || !x.isIgnored())
        .map(x -> x.buildParameter(typeSchemaPool, parameterIdExpressionHandler))
        .collect(toList());
  }

  private List<Parameter> buildNonStandardHeaders(TypeSchemaPool typeSchemaPool,
                                                  ParameterIdentifierExpressionHandler parameterIdExpressionHandler) {
    return nonStandardHeaderBuilders.stream()
        .filter(x -> x.isIgnored() == null || !x.isIgnored())
        .map(x -> x.buildParameter(typeSchemaPool, parameterIdExpressionHandler))
        .collect(toList());
  }

  // Security scheme types are not meant to be usable, except in that corner case where the user wants to use a custom security
  // type by declaring in the api spec an unsupported security type and in the descriptor referencing that security scheme with a
  // CUSTOM statement
  public void setType(
                      ConnectorSecurityScheme.SecuritySchemeType type) {
    if (!type.equals(CUSTOM_AUTHENTICATION) && !this.type.equals(UNSECURED)) {
      throw new IllegalStateException("\n"
          + "this method can only be called if the received security type is CUSTOM_AUTHENTICATION and the class type is UNSECURED. this is a bug");
    }
    this.type = type;
  }

}
