/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.connectormodel;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.JsonNodeFactory;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.fasterxml.jackson.databind.node.ValueNode;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.schema.TypeSchema;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.TreeMap;

public class TypeSchemaPool {

  private static final ObjectMapper MAPPER = new ObjectMapper();

  private final Map<String, TypeSchema> innerSchemaPool = new HashMap<>();

  public TypeSchema getUniqueTypeSchema(TypeSchema typeSchema) {
    String rawSchema = typeSchema.getRawSchema();
    try {
      rawSchema = getCanonicalJson(rawSchema);
    } catch (IOException e) {
      // not json
    }
    return innerSchemaPool.computeIfAbsent(rawSchema, str -> typeSchema);
  }

  /**
   * Returns a canonical version of the input JSON string.
   *
   * @param json the input JSON string
   * @return the canonical version of the input JSON string
   */
  static String getCanonicalJson(String json) throws IOException {
    JsonNode jsonNode = MAPPER.readTree(json);
    jsonNode = canonicalize(jsonNode, new JsonNodeFactory(true));
    return MAPPER.writeValueAsString(jsonNode);
  }

  /**
   * Processes a JSON tree sorting fields in every object in the tree.
   *
   * @param jsonNode an input JSON tree
   * @param nf a node factory
   * @return a newly created JSON tree, semantically equivalent to the input tree
   */
  private static JsonNode canonicalize(JsonNode jsonNode, JsonNodeFactory nf) {
    if (jsonNode instanceof ObjectNode) {
      Map<String, JsonNode> k = new TreeMap<>();
      jsonNode.fields().forEachRemaining(e -> k.put(e.getKey(), canonicalize(e.getValue(), nf)));
      return new ObjectNode(nf, k);
    } else if (jsonNode instanceof ArrayNode) {
      ArrayNode arrayNode = new ArrayNode(nf, jsonNode.size());
      jsonNode.elements().forEachRemaining(el -> arrayNode.add(canonicalize(el, nf)));
      return arrayNode;
    } else if (jsonNode instanceof ValueNode) {
      return jsonNode;
    }
    throw new RuntimeException("node " + jsonNode + " (" + jsonNode.getClass() + ") not supported");
  }
}
