/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.connectormodel.loader.descriptor;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.loader.descriptor.DescriptorTestConnectionLoader.loadTestConnection;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.loader.descriptor.DescriptorTypeLoader.loadDataType;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.HEADER;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.QUERY;
import static com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.DescriptorSecurityKind.CUSTOM;
import static com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.DescriptorSecurityKind.MOCKED;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.ConnectorModelBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.ParameterBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.SecurityDefaultParameterBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.SecurityDefaultParametersBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.SecuritySchemeBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.ConnectorSecurityScheme.SecuritySchemeType;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.BasicSecurityDefaultParametersDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.BearerSecurityDefaultParametersDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.DefaultParameters;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.DigestSecurityDefaultParametersDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.WeaveExpressionDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.CustomSecuritySchemeDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ParameterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.SecuritySchemeBaseDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.SecuritySchemeDescriptor;

import java.util.List;

public class DescriptorSecuritySchemeLoader {

  static ConnectorModelBuilder loadSecuritySchemes(ConnectorDescriptor connectorDescriptor,
                                                   ConnectorModelBuilder connectorModelBuilder) {
    for (SecuritySchemeBaseDescriptor securityDescriptor : connectorDescriptor.getSecurity()) {
      if (securityDescriptor.getKind().equals(CUSTOM)) {
        SecuritySchemeBuilder securitySchemeBuilder =
            connectorModelBuilder.getOrCreateSecuritySchemeBuilder(securityDescriptor.getName(),
                                                                   SecuritySchemeType.CUSTOM_AUTHENTICATION);
        securitySchemeBuilder.fqn(((CustomSecuritySchemeDescriptor) securityDescriptor).getFqn());
      } else if (securityDescriptor.getKind().equals(MOCKED)) {
        connectorModelBuilder.getOrCreateSecuritySchemeBuilder(securityDescriptor.getName(),
                                                               SecuritySchemeType.MOCKED);
      } else {
        loadSecurityScheme((SecuritySchemeDescriptor) securityDescriptor,
                           connectorModelBuilder.getOrCreateSecuritySchemeBuilder(securityDescriptor.getName(), null));
      }
    }

    return connectorModelBuilder;
  }

  private static void loadSecurityScheme(SecuritySchemeDescriptor securityDescriptor,
                                         SecuritySchemeBuilder securitySchemeBuilder) {

    securitySchemeBuilder
        .ignored(securityDescriptor.isIgnored())
        .alias(securityDescriptor.getAlias())
        .refined(securityDescriptor.getRefined())
        .testConnectionConfig(loadTestConnection(securityDescriptor.getTestConnection()));

    WeaveExpressionDescriptor refreshTokenExpression = securityDescriptor.getRefreshTokenConditionExpressionDescriptor();
    if (refreshTokenExpression != null) {
      securitySchemeBuilder.refreshTokenConditionExpression(refreshTokenExpression.getExpression());
    }

    loadSecurityParameters(securityDescriptor.getQueryParameters(), QUERY, securitySchemeBuilder);
    loadSecurityParameters(securityDescriptor.getHeaders(), HEADER, securitySchemeBuilder);
    loadDefaultParameters(securityDescriptor.getDefaultParameters(), securitySchemeBuilder.getSecurityDefaultParametersBuilder());
  }

  private static void loadSecurityParameters(List<ParameterDescriptor> parameters,
                                             ParameterType parameterType,
                                             SecuritySchemeBuilder securitySchemeBuilder) {
    if (parameters == null) {
      return;
    }

    for (ParameterDescriptor parameterDescriptor : parameters) {
      loadSecurityParameter(parameterDescriptor,
                            securitySchemeBuilder.getOrCreateParameterBuilder(parameterType, parameterDescriptor.getParamName()));
    }
  }

  private static void loadSecurityParameter(ParameterDescriptor parameterDescriptor,
                                            ParameterBuilder parameterBuilder) {


    parameterBuilder
        .displayName(parameterDescriptor.getDisplayName())
        .description(parameterDescriptor.getDescription())
        .defaultValue(parameterDescriptor.getDefaultValue())
        .required(parameterDescriptor.getRequired())
        .ignored(parameterDescriptor.isIgnored());

    loadDataType(parameterDescriptor, parameterBuilder.getTypeDefinitionBuilder());
  }

  private static void loadDefaultParameters(DefaultParameters defaultParameters,
                                            SecurityDefaultParametersBuilder securityDefaultParametersBuilder) {

    if (defaultParameters instanceof BasicSecurityDefaultParametersDescriptor) {
      BasicSecurityDefaultParametersDescriptor basic = ((BasicSecurityDefaultParametersDescriptor) defaultParameters);
      SecurityDefaultParametersBuilder.BasicAuthBuilder basicParametersBuilder = securityDefaultParametersBuilder.basic();

      if (basic.getUsername() != null) {
        basicParametersBuilder.username(new SecurityDefaultParameterBuilder()
            .displayName(basic.getUsername().getDisplayName())
            .description(basic.getUsername().getDescription()));
      }

      if (basic.getPassword() != null) {
        basicParametersBuilder.password(new SecurityDefaultParameterBuilder()
            .displayName(basic.getPassword().getDisplayName())
            .description(basic.getPassword().getDescription()));
      }
    }

    else if (defaultParameters instanceof BearerSecurityDefaultParametersDescriptor) {
      BearerSecurityDefaultParametersDescriptor bearer = ((BearerSecurityDefaultParametersDescriptor) defaultParameters);

      if (bearer.getToken() != null) {
        securityDefaultParametersBuilder.bearer()
            .token(new SecurityDefaultParameterBuilder()
                .displayName(bearer.getToken().getDisplayName())
                .description(bearer.getToken().getDescription()));
      }
    }

    else if (defaultParameters instanceof DigestSecurityDefaultParametersDescriptor) {
      DigestSecurityDefaultParametersDescriptor digest = ((DigestSecurityDefaultParametersDescriptor) defaultParameters);
      SecurityDefaultParametersBuilder.DigestAuthBuilder digestParametersBuilder = securityDefaultParametersBuilder.digest();

      if (digest.getUsername() != null) {
        digestParametersBuilder
            .username(new SecurityDefaultParameterBuilder()
                .displayName(digest.getUsername().getDisplayName())
                .description(digest.getUsername().getDescription()));
      }

      if (digest.getPassword() != null) {
        digestParametersBuilder
            .password(new SecurityDefaultParameterBuilder()
                .displayName(digest.getPassword().getDisplayName())
                .description(digest.getPassword().getDescription()));
      }
    }
  }
}

