/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder;

import static java.util.Objects.requireNonNull;
import static java.util.stream.Collectors.toList;
import static javax.ws.rs.core.MediaType.APPLICATION_JSON_TYPE;
import static javax.ws.rs.core.MediaType.APPLICATION_XML_TYPE;
import static org.apache.commons.lang3.ObjectUtils.defaultIfNull;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.TypeSchemaPool;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.ArrayTypeDefinition;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.EmptyTypeDefinition;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.MultipartTypeDefinition;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.ObjectTypeDefinition;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.PrimitiveTypeDefinition;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.TypeDefinition;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.UnionTypeDefinition;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.schema.TypeSchema;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.schema.XmlTypeSchema;

import java.util.List;
import java.util.function.Supplier;

import javax.ws.rs.core.MediaType;

public class TypeDefinitionBuilder {

  private Class<?> classToBuild;
  private TypeDefinitionBuilder innerType;
  private List<ParameterBuilder> parts;
  private PrimitiveTypeDefinition.PrimitiveType primitiveType;
  private List<TypeDefinitionBuilder> unionTypes;
  private MediaType mediaType;
  private String example;
  private List<String> enumValues;
  private String displayName;
  private String description;
  private Supplier<String> rawSchemaSupplier;
  private String schemaPath;
  private String elementName;

  public TypeDefinitionBuilder primitive(PrimitiveTypeDefinition.PrimitiveType primitiveType) {
    requireNonNull(primitiveType);

    this.classToBuild = PrimitiveTypeDefinition.class;
    this.primitiveType = primitiveType;
    return this;
  }

  public TypeDefinitionBuilder array(TypeDefinitionBuilder innerType) {
    requireNonNull(innerType);

    this.classToBuild = ArrayTypeDefinition.class;
    this.innerType = innerType;
    return this;
  }

  public TypeDefinitionBuilder multipart(List<ParameterBuilder> parts) {
    requireNonNull(parts);

    this.classToBuild = MultipartTypeDefinition.class;
    this.parts = parts;
    return this;
  }

  public ParameterBuilder getPartParameter(String externalName) {
    return parts.stream()
        .filter(x -> x.getExternalName().equals(externalName))
        .findFirst()
        .orElseThrow(() -> new IllegalArgumentException("Specified part parameter does not exist. This is a bug."));
  }

  public TypeDefinitionBuilder object() {
    this.classToBuild = ObjectTypeDefinition.class;
    return this;
  }

  public TypeDefinitionBuilder union(List<TypeDefinitionBuilder> unionTypes) {
    requireNonNull(unionTypes);

    this.classToBuild = UnionTypeDefinition.class;
    this.unionTypes = unionTypes;
    return this;
  }

  public TypeDefinitionBuilder empty() {
    this.classToBuild = EmptyTypeDefinition.class;
    return this;
  }

  public TypeDefinitionBuilder displayName(String displayName) {
    this.displayName = defaultIfNull(displayName, this.displayName);
    return this;
  }

  public TypeDefinitionBuilder description(String description) {
    this.description = defaultIfNull(description, this.description);
    return this;
  }

  public TypeDefinitionBuilder mediaType(MediaType mediaType) {
    this.mediaType = defaultIfNull(mediaType, this.mediaType);
    return this;
  }

  public TypeDefinitionBuilder typeSchema(Supplier<String> rawSchema) {
    this.rawSchemaSupplier = defaultIfNull(rawSchema, this.rawSchemaSupplier);
    return this;
  }

  public TypeDefinitionBuilder enumValues(List<String> enumValues) {
    this.enumValues = defaultIfNull(enumValues, this.enumValues);
    return this;
  }

  public TypeDefinitionBuilder example(String example) {
    this.example = defaultIfNull(example, this.example);
    return this;
  }

  public TypeDefinitionBuilder schemaPath(String schemaPath) {
    this.schemaPath = schemaPath;
    return this;
  }

  public TypeDefinitionBuilder elementName(String elementName) {
    this.elementName = elementName;
    return this;
  }

  public boolean canBuild() {
    return classToBuild != null;
  }

  public boolean isMultipart() {
    return this.classToBuild == MultipartTypeDefinition.class;
  }

  public TypeDefinition build(TypeSchemaPool typeSchemaPool, String forcedSchema) {
    requireNonNull(classToBuild, "Must select a type. This is a bug.");

    TypeSchema typeSchema = buildTypeSchema(typeSchemaPool, forcedSchema);
    MediaType mediaType = buildMediaType(typeSchema);

    if (classToBuild.equals(ObjectTypeDefinition.class) || isNotBlank(forcedSchema)) {
      return new ObjectTypeDefinition(mediaType, example, enumValues, typeSchema,
                                      displayName, description);
    }

    else if (classToBuild.equals(ArrayTypeDefinition.class)) {
      return new ArrayTypeDefinition(mediaType, example, enumValues, typeSchema,
                                     innerType.mediaType(mediaType).build(typeSchemaPool, null),
                                     displayName, description);
    }

    else if (classToBuild.equals(MultipartTypeDefinition.class)) {
      return new MultipartTypeDefinition(example, enumValues, typeSchema,
                                         parts.stream().map(x -> x.buildPartParameter(typeSchemaPool)).collect(toList()),
                                         displayName, description);
    }

    else if (classToBuild.equals(PrimitiveTypeDefinition.class)) {
      return new PrimitiveTypeDefinition(mediaType, example, enumValues, typeSchema,
                                         primitiveType, displayName, description);
    }

    else if (classToBuild.equals(UnionTypeDefinition.class)) {
      return new UnionTypeDefinition(mediaType, example, enumValues, typeSchema,
                                     unionTypes.stream().map(x -> x.build(typeSchemaPool, null)).collect(toList()),
                                     displayName, description);
    }

    else if (classToBuild.equals(EmptyTypeDefinition.class)) {
      return new EmptyTypeDefinition(mediaType, example, enumValues, displayName, description);
    }

    throw new IllegalArgumentException("");
  }

  private MediaType buildMediaType(TypeSchema typeSchema) {
    if (mediaType != null) {
      return mediaType;
    }

    if (typeSchema != null) {
      return (typeSchema instanceof XmlTypeSchema) ? APPLICATION_XML_TYPE : APPLICATION_JSON_TYPE;
    }

    return null;
  }

  private TypeSchema buildTypeSchema(TypeSchemaPool typeSchemaPool, String forcedTypeSchema) {
    String rawSchema =
        isNotBlank(forcedTypeSchema) ? forcedTypeSchema : (rawSchemaSupplier != null ? rawSchemaSupplier.get() : null);

    return isNotBlank(rawSchema) ? new TypeSchemaBuilder(typeSchemaPool)
        .buildTypeSchema(rawSchema, elementName, schemaPath) : null;
  }
}
