/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.connectormodel.loader.api;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.loader.api.ApiParameterLoader.loadParameter;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.PART;
import static java.util.stream.Collectors.toList;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.OperationBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.ParameterBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.TypeDefinitionBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.PrimitiveTypeDefinition;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.type.APIPrimitiveTypeModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.type.APITypeModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.type.schema.APITypeSchemaModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.type.schema.APIXmlTypeSchemaModel;

import java.util.List;

public class ApiTypeLoader {

  public static OperationBuilder loadInputTypeDefinitions(List<APITypeModel> apiTypeModels,
                                                          OperationBuilder connectorOperationBuilder) {
    for (APITypeModel apiTypeModel : apiTypeModels) {
      TypeDefinitionBuilder inputMetadataBuilder =
          connectorOperationBuilder.getInputMetadataBuilder(apiTypeModel.getMediaType());

      loadTypeDefinition(apiTypeModel, inputMetadataBuilder);
    }
    return connectorOperationBuilder;
  }

  public static OperationBuilder loadOutputTypeDefinitions(List<APITypeModel> apiTypeModels,
                                                           OperationBuilder connectorOperationBuilder) {
    for (APITypeModel apiTypeModel : apiTypeModels) {
      TypeDefinitionBuilder outputMetadataBuilder =
          connectorOperationBuilder.getOutputMetadataBuilder(apiTypeModel.getMediaType());

      loadTypeDefinition(apiTypeModel, outputMetadataBuilder);
    }
    return connectorOperationBuilder;
  }

  static TypeDefinitionBuilder loadTypeDefinition(APITypeModel apiTypeModel,
                                                  TypeDefinitionBuilder typeDefinitionBuilder) {
    typeDefinitionBuilder
        .displayName(apiTypeModel.getDisplayName())
        .description(apiTypeModel.getDescription())
        .enumValues(apiTypeModel.getEnumValues())
        .example(apiTypeModel.getExample())
        .mediaType(apiTypeModel.getMediaType());

    APITypeSchemaModel typeSchema = apiTypeModel.getAPITypeSchemaModel();
    if (typeSchema != null) {
      typeDefinitionBuilder.typeSchema(typeSchema.getSchemaSupplier());

      if (typeSchema instanceof APIXmlTypeSchemaModel) {
        typeDefinitionBuilder.elementName(((APIXmlTypeSchemaModel) typeSchema).getElementName());
        typeDefinitionBuilder.schemaPath(((APIXmlTypeSchemaModel) typeSchema).getPath());
      }
    }

    if (apiTypeModel.getApiType() == null) {
      return typeDefinitionBuilder;
    }

    switch (apiTypeModel.getApiType()) {
      case ARRAY:
        typeDefinitionBuilder.array(loadTypeDefinition(apiTypeModel.getInnerType(), new TypeDefinitionBuilder()));
        break;
      case EMPTY:
        typeDefinitionBuilder.empty();
        break;
      case MULTIPART:
        List<ParameterBuilder> partParameterBuilders =
            apiTypeModel.getParts().stream()
                .map(x -> loadParameter(x, new ParameterBuilder(PART, x.getExternalName())))
                .collect(toList());
        typeDefinitionBuilder.multipart(partParameterBuilders);
        break;
      case PRIMITIVE:
        typeDefinitionBuilder.primitive(buildPrimitiveType(apiTypeModel.getPrimitiveTypeModel()));
        break;
      case UNION_TYPE:
        List<TypeDefinitionBuilder> unionTypeBuilders =
            apiTypeModel.getUnionTypes().stream()
                .map(x -> loadTypeDefinition(x, new TypeDefinitionBuilder()))
                .collect(toList());
        typeDefinitionBuilder.union(unionTypeBuilders);
        break;
      case OBJECT_TYPE:
        typeDefinitionBuilder.object();
        break;
      default:
        throw new IllegalArgumentException("Type not supported. This is a bug.");
    }

    return typeDefinitionBuilder;
  }

  private static PrimitiveTypeDefinition.PrimitiveType buildPrimitiveType(APIPrimitiveTypeModel primitiveTypeModel) {
    switch (primitiveTypeModel.getPrimitiveType()) {
      case BOOLEAN:
        return PrimitiveTypeDefinition.PrimitiveType.BOOLEAN;
      case DATE:
        return PrimitiveTypeDefinition.PrimitiveType.DATE;
      case DATE_ONLY:
        return PrimitiveTypeDefinition.PrimitiveType.DATE_ONLY;
      case DATE_TIME_ONLY:
        return PrimitiveTypeDefinition.PrimitiveType.DATE_TIME_ONLY;
      case DATE_TIME:
        return PrimitiveTypeDefinition.PrimitiveType.DATE_TIME;
      case FILE:
        return PrimitiveTypeDefinition.PrimitiveType.FILE;
      case NUMBER:
        return PrimitiveTypeDefinition.PrimitiveType.NUMBER;
      case INTEGER:
        return PrimitiveTypeDefinition.PrimitiveType.INTEGER;
      case STRING:
        return PrimitiveTypeDefinition.PrimitiveType.STRING;
      case TIME_ONLY:
        return PrimitiveTypeDefinition.PrimitiveType.TIME_ONLY;
      default:
        throw new IllegalArgumentException("Primitive Type not supported. This is a bug.");
    }
  }
}
