/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.connectormodel.loader.descriptor;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.loader.descriptor.DescriptorParameterLoader.loadParameters;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.loader.descriptor.DescriptorSampleDataLoader.loadSampleData;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.loader.descriptor.DescriptorTypeLoader.loadMediaType;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.HEADER;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.QUERY;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.URI;
import static org.apache.commons.lang3.ObjectUtils.defaultIfNull;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.QueryParamArrayFormat;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.ConnectorModelBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.OperationBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.ParameterBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.TypeDefinitionBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.EndPointDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.PartDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.RequestDescriptor;

import org.apache.commons.lang3.StringUtils;

public class DescriptorOperationLoader {

  static ConnectorModelBuilder loadOperations(ConnectorDescriptor descriptor, ConnectorModelBuilder connectorModelBuilder) {

    for (EndPointDescriptor endPointDescriptor : descriptor.getEndpoints()) {
      loadEndpoint(endPointDescriptor, connectorModelBuilder);
      for (OperationDescriptor operationDescriptor : endPointDescriptor.getOperations()) {
        OperationBuilder operationBuilder =
            connectorModelBuilder.getOperationBuilder(null, endPointDescriptor.getPath(), operationDescriptor.getMethod(), null);

        loadOperation(descriptor, endPointDescriptor, operationDescriptor, operationBuilder);
      }
    }

    return connectorModelBuilder;
  }

  private static ConnectorModelBuilder loadEndpoint(EndPointDescriptor endPointDescriptor,
                                                    ConnectorModelBuilder connectorModelBuilder) {

    for (OperationBuilder operationBuilder : connectorModelBuilder.getOperationBuildersByPath(endPointDescriptor.getPath())) {
      operationBuilder.ignored(endPointDescriptor.isIgnored());
      operationBuilder.alternativeBaseUri(endPointDescriptor.getBaseUri());
    }

    return connectorModelBuilder;
  }

  private static OperationBuilder loadOperation(ConnectorDescriptor connectorDescriptor,
                                                EndPointDescriptor endPointDescriptor,
                                                OperationDescriptor operationDescriptor,
                                                OperationBuilder operationBuilder) {

    operationBuilder
        .displayName(operationDescriptor.getDisplayName())
        .description(operationDescriptor.getDescription())
        .alternativeBaseUri(loadAlternativeBaseUri(endPointDescriptor, operationDescriptor))
        .pagination(operationDescriptor.getPagination())
        .skipOutputTypeValidation(operationDescriptor.getSkipOutputTypeValidation())
        .isVoidOperation(operationDescriptor.getVoidOperation())
        .queryParamArrayFormat(loadOperationQueryParamArrayFormat(connectorDescriptor, operationDescriptor))
        .ignored(loadIgnoredOperation(endPointDescriptor, operationDescriptor));

    loadOperationInputType(operationDescriptor, operationBuilder);
    loadOperationOutputType(operationDescriptor, operationBuilder);

    if (operationDescriptor.getExpects() != null) {
      loadParameters(operationDescriptor.getExpects().getUriParameter(), URI, operationBuilder);
      loadParameters(operationDescriptor.getExpects().getQueryParameter(), QUERY, operationBuilder);
      loadParameters(operationDescriptor.getExpects().getHeader(), HEADER, operationBuilder);
    }

    loadSampleData(operationDescriptor.getSampleDataExpressionDescriptor(), operationBuilder.getSampleDataBuilder());

    return operationBuilder;
  }

  static OperationBuilder loadOperationInputType(OperationDescriptor operationDescriptor,
                                                 OperationBuilder operationBuilder) {


    if (isNotBlank(operationDescriptor.getInputMediaType())) {
      operationBuilder.setDefaultInputMediaType(loadMediaType(operationDescriptor.getInputMediaType()));
    }

    if (isNotBlank(operationDescriptor.getInputTypeSchema())) {
      operationBuilder.forceInputTypeSchema(operationDescriptor.getInputTypeSchema());
    }

    loadParts(operationDescriptor.getExpects(), operationBuilder);

    return operationBuilder;
  }

  private static void loadParts(RequestDescriptor expects, OperationBuilder operationBuilder) {

    for (TypeDefinitionBuilder builder : operationBuilder.getMultipartInputMetadataBuilders()) {
      for (PartDescriptor partDescriptor : expects.getPart()) {

        ParameterBuilder partParameter = builder.getPartParameter(partDescriptor.getPartName());

        partParameter
            .displayName(partDescriptor.getDisplayName())
            .description(partDescriptor.getDescription())
            .filePart(partDescriptor.isFilePart());

        TypeDefinitionBuilder typeDefinitionBuilder = partParameter.getTypeDefinitionBuilder();

        if (StringUtils.isNotBlank(partDescriptor.getInputType())) {
          typeDefinitionBuilder.object().typeSchema(partDescriptor::getInputType);
        }

        typeDefinitionBuilder.mediaType(loadMediaType(partDescriptor.getContentType()));
      }
    }
  }

  static OperationBuilder loadOperationOutputType(OperationDescriptor operationDescriptor,
                                                  OperationBuilder operationBuilder) {

    if (isNotBlank(operationDescriptor.getOutputMediaType())) {
      operationBuilder.setDefaultOutputMediaType(loadMediaType(operationDescriptor.getOutputMediaType()));
    }

    if (isNotBlank(operationDescriptor.getOutputTypeSchema())) {
      operationBuilder.forceOutputTypeSchema(operationDescriptor.getOutputTypeSchema());
    }

    return operationBuilder;
  }

  private static Boolean loadIgnoredOperation(EndPointDescriptor endPointDescriptor,
                                              OperationDescriptor operationDescriptor) {
    return defaultIfNull(operationDescriptor.isIgnored(), endPointDescriptor.isIgnored());
  }

  private static QueryParamArrayFormat loadOperationQueryParamArrayFormat(ConnectorDescriptor connectorDescriptor,
                                                                          OperationDescriptor operationDescriptor) {

    if (isNotBlank(operationDescriptor.getQueryParamArrayFormat())) {
      return QueryParamArrayFormat.valueOf(operationDescriptor.getQueryParamArrayFormat().toUpperCase());
    }

    else if (isNotBlank(connectorDescriptor.getQueryParamArrayFormat())) {
      return QueryParamArrayFormat.valueOf(connectorDescriptor.getQueryParamArrayFormat().toUpperCase());
    }

    return null;
  }

  private static String loadAlternativeBaseUri(EndPointDescriptor endPointDescriptor, OperationDescriptor operationDescriptor) {
    return isNotBlank(operationDescriptor.getBaseUri()) ? operationDescriptor.getBaseUri()
        : (isNotBlank(endPointDescriptor.getBaseUri()) ? endPointDescriptor.getBaseUri() : null);
  }
}
