/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder;

import static java.util.Objects.requireNonNull;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.ObjectUtils.defaultIfNull;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorOperation;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.HTTPMethod;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.TypeSchemaPool;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.resolver.sampledata.SampleDataBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.generic.ParameterDataType;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.trigger.Trigger;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.TypeDefinition;

import java.util.ArrayList;
import java.util.List;

import javax.ws.rs.core.MediaType;

public class TriggerBuilder {

  private final String name;

  private String path;
  private HTTPMethod method;
  private String displayName;
  private String description;
  private String itemsExpression;
  private String identityExpression;
  private String requestBodyExpression;
  private String watermarkExpression;
  private ParameterDataType watermarkType;
  private final List<TriggerParameterBuilder> parameterBuilders = new ArrayList<>();
  private final List<ParameterBindingBuilder> parameterBindingBuilders = new ArrayList<>();
  private String outputTypeSchema;
  private String outputMediaType;
  private final SampleDataBuilder sampleDataBuilder = new SampleDataBuilder();
  private Boolean ignored;

  TriggerBuilder(String name) {
    requireNonNull(name);

    this.name = name;
  }

  public String getName() {
    return name;
  }

  public String getPath() {
    return path;
  }

  public HTTPMethod getMethod() {
    return method;
  }

  public TriggerBuilder path(String path) {
    this.path = defaultIfNull(path, this.path);
    return this;
  }

  public TriggerBuilder method(HTTPMethod method) {
    this.method = defaultIfNull(method, this.method);
    return this;
  }

  public TriggerBuilder displayName(String displayName) {
    this.displayName = defaultIfNull(displayName, this.displayName);
    return this;
  }

  public String getDisplayName() {
    return displayName;
  }

  public TriggerBuilder setDescription(String description) {
    this.description = defaultIfNull(description, this.description);
    return this;
  }

  public TriggerBuilder itemsExpression(String itemsExpression) {
    this.itemsExpression = defaultIfNull(itemsExpression, this.itemsExpression);
    return this;
  }

  public String getItemsExpression() {
    return itemsExpression;
  }

  public TriggerBuilder identityExpression(String identityExpression) {
    this.identityExpression = defaultIfNull(identityExpression, this.identityExpression);
    return this;
  }

  public TriggerBuilder requestBodyExpression(String requestBodyExpression) {
    this.requestBodyExpression = defaultIfNull(requestBodyExpression, this.requestBodyExpression);
    return this;
  }

  public TriggerBuilder watermarkExpression(String watermarkExpression) {
    this.watermarkExpression = defaultIfNull(watermarkExpression, this.watermarkExpression);
    return this;
  }

  public TriggerBuilder watermarkType(ParameterDataType watermarkType) {
    this.watermarkType = defaultIfNull(watermarkType, this.watermarkType);
    return this;
  }

  public TriggerBuilder outputTypeSchema(String outputTypeSchema) {
    this.outputTypeSchema = defaultIfNull(outputTypeSchema, this.outputTypeSchema);
    return this;
  }

  public String getOutputTypeSchema() {
    return outputTypeSchema;
  }

  public TriggerBuilder outputMediaType(String outputMediaType) {
    this.outputMediaType = defaultIfNull(outputMediaType, this.outputMediaType);
    return this;
  }

  public TriggerBuilder ignored(Boolean ignored) {
    this.ignored = defaultIfNull(ignored, this.ignored);
    return this;
  }

  public ParameterBindingBuilder getOrCreateParameterBindingBuilder(String name, ParameterType parameterType) {
    ParameterBindingBuilder parameterBindingBuilder = parameterBindingBuilders.stream()
        .filter(x -> x.getName().equals(name) && x.getParameterType().equals(parameterType))
        .findFirst().orElse(null);

    if (parameterBindingBuilder == null) {
      parameterBindingBuilder = new ParameterBindingBuilder(name, parameterType);
      parameterBindingBuilders.add(parameterBindingBuilder);
    }

    return parameterBindingBuilder;
  }

  public TriggerParameterBuilder getOrCreateParameterBuilder(String name) {
    TriggerParameterBuilder parameterBuilder = parameterBuilders.stream()
        .filter(x -> x.getName().equalsIgnoreCase(name))
        .findFirst().orElse(null);

    if (parameterBuilder == null) {
      parameterBuilder = new TriggerParameterBuilder(name);
      parameterBuilders.add(parameterBuilder);
    }

    return parameterBuilder;
  }

  public SampleDataBuilder getSampleDataExpressionBuilder() {
    return sampleDataBuilder;
  }

  Trigger build(List<ConnectorOperation> operations, TypeSchemaPool typeSchemaPool) {

    ConnectorOperation operation = operations.stream()
        .filter(op -> op.getHttpMethod().equals(method) && op.getPath().equals(path)).findFirst()
        .orElseThrow(() -> new IllegalArgumentException("Invalid path/method for this trigger"));

    Trigger trigger = new Trigger(name,
                                  displayName,
                                  description,
                                  itemsExpression,
                                  identityExpression,
                                  requestBodyExpression,
                                  watermarkExpression,
                                  buildWatermarkType(watermarkType),
                                  parameterBuilders.stream().map(TriggerParameterBuilder::build).collect(toList()),
                                  parameterBindingBuilders.stream().map(ParameterBindingBuilder::build)
                                      .collect(toList()),
                                  getTriggerOutputType(operation, typeSchemaPool),
                                  operation);

    trigger.setSampleData(sampleDataBuilder.build(trigger));

    return trigger;
  }

  private ParameterDataType buildWatermarkType(ParameterDataType watermarkType) {
    if (watermarkType != null) {
      return watermarkType;
    }

    return ParameterDataType.STRING;
  }

  private TypeDefinition getTriggerOutputType(ConnectorOperation operation, TypeSchemaPool typeSchemaPool) {
    TypeDefinition triggerOutputType = null;

    if (isNotBlank(outputTypeSchema)) {
      TypeDefinitionBuilder typeBuilder = new TypeDefinitionBuilder().object();

      if (isNotBlank(outputMediaType)) {
        typeBuilder.mediaType(MediaType.valueOf(outputMediaType));
      } else {
        if (operation.getOutputMetadata() != null) {
          typeBuilder.mediaType(operation.getOutputMetadata().getMediaType());
        }
      }

      triggerOutputType = typeBuilder.build(typeSchemaPool, outputTypeSchema);
    }

    return triggerOutputType;
  }

  public boolean isIgnored() {
    return ignored != null && ignored;
  }
}
