/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.resolver.sampledata;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.HEADER;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.QUERY;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.URI;
import static java.lang.String.format;
import static java.util.Objects.requireNonNull;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.ObjectUtils.defaultIfNull;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorOperation;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.resolver.ResolverDefinitionBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dataexpression.httprequest.HttpRequestBinding;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dataexpression.httprequest.HttpRequestDataExpression;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dataexpression.script.DwScriptDataExpression;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dataexpression.script.ScriptDataExpression;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.expression.DataWeaveExpression;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.generic.Argument;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.Parameter;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.sampledata.SampleDataDefinition;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.trigger.Trigger;

import java.util.List;

public class SampleDataDefinitionBuilder extends ResolverDefinitionBuilder<SampleDataDefinition> {

  private HttpRequestDataExpression httpRequestDataExpression;
  private ScriptDataExpression scriptDataExpression;

  public SampleDataDefinitionBuilder httpRequestDataExpression(HttpRequestDataExpression httpRequestDataExpression) {
    this.httpRequestDataExpression = defaultIfNull(httpRequestDataExpression, this.httpRequestDataExpression);
    return this;
  }

  public SampleDataDefinitionBuilder scriptDataExpression(ScriptDataExpression scriptDataExpression) {
    this.scriptDataExpression = defaultIfNull(scriptDataExpression, this.scriptDataExpression);
    return this;
  }

  public SampleDataDefinition build() {
    requireNonNull(httpRequestDataExpression);

    return new SampleDataDefinition(httpRequestDataExpression, scriptDataExpression);
  }

  public SampleDataDefinition buildSameOperation(ConnectorOperation operation) {
    requireNonNull(operation);

    return buildSampleDataSameOperationDefinition(operation);
  }

  public SampleDataDefinition buildSameOperation(Trigger trigger) {
    requireNonNull(trigger);

    return buildSampleDataSameOperationDefinition(trigger);
  }

  private SampleDataDefinition buildSampleDataSameOperationDefinition(ConnectorOperation connectorOperation) {
    DwScriptDataExpression paginationExpression = null;
    if (connectorOperation.hasPagination()) {
      paginationExpression = new DwScriptDataExpression(connectorOperation.getPagination().getPaginationResponseExpression());
    }

    return new SampleDataDefinition(new HttpRequestDataExpression(connectorOperation.getPath(),
                                                                  connectorOperation.getHttpMethod(),
                                                                  buildBindings(connectorOperation)),
                                    paginationExpression);
  }

  private SampleDataDefinition buildSampleDataSameOperationDefinition(Trigger trigger) {
    return new SampleDataDefinition(new HttpRequestDataExpression(trigger.getOperation().getPath(),
                                                                  trigger.getOperation().getHttpMethod(),
                                                                  buildTriggerBindings(trigger)),
                                    new DwScriptDataExpression(getExpressionForFirstItem(trigger.getItemsExpression())));
  }

  /**
   * Modifies an expression that returns an array and makes it return it's first element.
   *
   * i.e:
   *
   * #[payload.items] -> (payload.items)[0].
   *
   * #[payload.data default []] -> (payload.data default [])[0].
   *
   * payload.data default [] -> (payload.data default [])[0].
   *
   */
  private String getExpressionForFirstItem(String expression) {
    if (expression.startsWith("#[") && expression.endsWith("]")) {
      expression = expression.substring(2, expression.length() - 1);
    }

    return "(" + expression + ")[0]";
  }

  private HttpRequestBinding buildTriggerBindings(Trigger trigger) {
    if (trigger.getParameterBindings() == null || trigger.getParameterBindings().isEmpty()) {
      return null;
    }

    return new HttpRequestBinding(buildTriggerArguments(trigger, URI),
                                  buildTriggerArguments(trigger, QUERY),
                                  buildTriggerArguments(trigger, HEADER));
  }

  private List<Argument> buildTriggerArguments(Trigger trigger, ParameterType parameterType) {
    return trigger.getParameterBindings().stream()
        .filter(x -> x.getParameterType().equals(parameterType))
        .map(x -> buildArgument(x.getName(), x.getExpression()))
        .collect(toList());
  }

  private HttpRequestBinding buildBindings(ConnectorOperation connectorOperation) {
    List<Parameter> requiredUriParams = getRequiredParameters(connectorOperation.getUriParameters());
    List<Parameter> requiredQueryParams = getRequiredParameters(connectorOperation.getQueryParameters());
    List<Parameter> requiredHeaders = getRequiredParameters(connectorOperation.getHeaders());

    if (requiredUriParams.isEmpty() && requiredQueryParams.isEmpty() && requiredHeaders.isEmpty()) {
      return null;
    }

    return new HttpRequestBinding(buildArguments(requiredUriParams, URI),
                                  buildArguments(requiredQueryParams, QUERY),
                                  buildArguments(requiredHeaders, HEADER));
  }

  private List<Parameter> getRequiredParameters(List<Parameter> parameterList) {
    return parameterList.stream().filter(Parameter::isRequired).collect(toList());
  }

  private List<Argument> buildArguments(List<Parameter> parameters, ParameterType parameterType) {
    return parameters.stream().map(x -> buildArgument(x.getExternalName(), parameterType)).collect(toList());
  }

  private Argument buildArgument(String externalName, ParameterType parameterType) {
    return buildArgument(externalName, format("%s.%s", parameterType.getBinding(), externalName));
  }

  private Argument buildArgument(String externalName, String expression) {
    return new Argument(externalName, new DataWeaveExpression(expression));
  }
}
