/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.Protocol.HTTP;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.Protocol.HTTPS;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.QueryParamArrayFormat.MULTIMAP;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.ConnectorSecurityScheme.SecuritySchemeType.UNSECURED;
import static com.mulesoft.connectivity.rest.sdk.internal.webapi.util.XmlUtils.getXmlName;
import static com.mulesoft.connectivity.rest.sdk.internal.webapi.util.XmlUtils.removeMavenArtifactUnwantedCharacters;
import static com.mulesoft.connectivity.rest.sdk.internal.webapi.util.XmlUtils.removeMavenGroupUnwantedCharacters;
import static com.mulesoft.connectivity.rest.sdk.internal.webapi.util.XmlUtils.removeMavenVersionUnwantedCharacters;
import static java.lang.String.format;
import static java.util.Arrays.asList;
import static java.util.Objects.requireNonNull;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.ObjectUtils.defaultIfNull;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

import org.mule.weave.v2.parser.phase.CompilationException;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorCategory;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorConfiguration;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.operation.ConnectorOperation;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorPackage;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.Protocol;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.QueryParamArrayFormat;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.TypeSchemaPool;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dw.BodyIdentifierExpressionHandler;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dw.OperationDisplayNameExpressionHandler;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dw.OperationIdentifierExpressionHandler;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dw.ParameterIdentifierExpressionHandler;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.pagination.Pagination;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.ConnectorSecurityScheme;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.TestConnectionConfig;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.trigger.Trigger;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.BaseEndpointDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.exception.ModelGenerationException;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import javax.ws.rs.core.MediaType;

import org.apache.commons.lang3.StringUtils;

public class ConnectorModelBuilder {

  private final static Protocol[] DEFAULT_PROTOCOLS = {HTTP, HTTPS};

  private String connectorName;
  private String connectorDescription;
  private String mavenGroup;
  private String mavenArtifactId;
  private String mavenVersion;
  private String baseJavaPackage;
  private ConnectorCategory connectorCategory;
  private List<Protocol> supportedProtocols;
  private String extensionXml;
  private QueryParamArrayFormat queryParamArrayFormat;
  private final List<PaginationBuilder> paginationBuilders = new ArrayList<>();
  private final List<OperationBuilder> operationBuilders = new ArrayList<>();
  private final List<TriggerBuilder> triggerBuilders = new ArrayList<>();
  private final List<SecuritySchemeBuilder> securitySchemeBuilders = new ArrayList<>();
  private final BaseUriBuilder baseUriBuilder = new BaseUriBuilder();
  private Boolean skipOutputTypeValidation;
  private String operationIdentifierExpression;
  private String operationDisplayNameExpression;
  private String parameterIdentifierExpression;
  private String bodyIdentifierExpression;
  private MediaType defaultInputMediaType;
  private MediaType defaultOutputMediaType;
  private TestConnectionConfig testConnectionConfig;
  private ConnectorConfiguration connectorConfiguration;
  private final List<InterceptorBuilder> interceptorBuilders = new ArrayList<>();

  public ConnectorModelBuilder name(String connectorName) {
    this.connectorName = defaultIfNull(connectorName, this.connectorName);
    return this;
  }

  public ConnectorModelBuilder description(String connectorDescription) {
    this.connectorDescription = defaultIfNull(connectorDescription, this.connectorDescription);;
    return this;
  }

  public ConnectorModelBuilder mavenGroup(String mavenGroup) {
    this.mavenGroup = defaultIfNull(mavenGroup, this.mavenGroup);;
    return this;
  }

  public ConnectorModelBuilder mavenArtifactId(String mavenArtifactId) {
    this.mavenArtifactId = defaultIfNull(mavenArtifactId, this.mavenArtifactId);;
    return this;
  }

  public ConnectorModelBuilder mavenVersion(String mavenVersion) {
    this.mavenVersion = defaultIfNull(mavenVersion, this.mavenVersion);;
    return this;
  }

  public ConnectorModelBuilder baseJavaPackage(String baseJavaPackage) {
    this.baseJavaPackage = defaultIfNull(baseJavaPackage, this.baseJavaPackage);;
    return this;
  }

  public ConnectorModelBuilder category(ConnectorCategory connectorCategory) {
    this.connectorCategory = defaultIfNull(connectorCategory, this.connectorCategory);;
    return this;
  }

  public ConnectorModelBuilder supportedProtocols(List<Protocol> protocols) {
    this.supportedProtocols = defaultIfNull(protocols, this.supportedProtocols);;
    return this;
  }

  public ConnectorModelBuilder queryParamArrayFormat(QueryParamArrayFormat queryParamArrayFormat) {
    this.queryParamArrayFormat = defaultIfNull(queryParamArrayFormat, this.queryParamArrayFormat);;
    return this;
  }

  public ConnectorModelBuilder extensionXml(String extensionXml) {
    this.extensionXml = defaultIfNull(extensionXml, this.extensionXml);;
    return this;
  }

  public BaseUriBuilder getBaseUriBuilder() {
    return this.baseUriBuilder;
  }

  public Boolean getSkipOutputTypeValidation() {
    return this.skipOutputTypeValidation;
  }

  public ConnectorModelBuilder skipOutputTypeValidation(Boolean skipOutputTypeValidation) {
    this.skipOutputTypeValidation = defaultIfNull(skipOutputTypeValidation, this.skipOutputTypeValidation);;
    return this;
  }

  public ConnectorModelBuilder operationDisplayNameExpression(String operationDisplayNameExpression) {
    this.operationDisplayNameExpression = defaultIfNull(operationDisplayNameExpression, this.operationDisplayNameExpression);;
    return this;
  }

  public ConnectorModelBuilder operationIdentifierExpression(String operationIdentifierExpression) {
    if (this.operationIdentifierExpression != null) {
      throw new IllegalArgumentException("operationIdentifierExpression cannot be overridden. This is a bug.");
    }
    this.operationIdentifierExpression = operationIdentifierExpression;
    return this;
  }

  public ConnectorModelBuilder parameterIdentifierExpression(String parameterIdentifierExpression) {
    if (this.parameterIdentifierExpression != null) {
      throw new IllegalArgumentException("parameterIdentifierExpression cannot be overridden. This is a bug.");
    }
    this.parameterIdentifierExpression = parameterIdentifierExpression;
    return this;
  }

  public ConnectorModelBuilder bodyIdentifierExpression(String bodyIdentifierExpression) {
    if (this.bodyIdentifierExpression != null) {
      throw new IllegalArgumentException("bodyIdentifierExpression cannot be overridden. This is a bug.");
    }
    this.bodyIdentifierExpression = bodyIdentifierExpression;
    return this;
  }

  public MediaType getDefaultInputMediaType() {
    return defaultInputMediaType;
  }

  public ConnectorModelBuilder defaultInputMediaType(MediaType defaultInputMediaType) {
    this.defaultInputMediaType = defaultInputMediaType;
    return this;
  }

  public MediaType getDefaultOutputMediaType() {
    return defaultOutputMediaType;
  }

  public ConnectorModelBuilder defaultOutputMediaType(MediaType defaultOutputMediaType) {
    this.defaultOutputMediaType = defaultOutputMediaType;
    return this;
  }

  public ConnectorModelBuilder testConnection(TestConnectionConfig testConnectionConfig) {
    this.testConnectionConfig = testConnectionConfig;
    return this;
  }

  public ConnectorModelBuilder connectorConfiguration(ConnectorConfiguration connectorConfiguration) {
    this.connectorConfiguration = connectorConfiguration;
    return this;
  }

  public List<OperationBuilder> getOperationBuilders() {
    return operationBuilders;
  }

  public List<OperationBuilder> getOperationBuildersByPath(String path) {
    return operationBuilders.stream()
        .filter(x -> x.getPath().equalsIgnoreCase(path))
        .collect(toList());
  }

  public Optional<OperationBuilder> getOperationBuildersByOperationId(BaseEndpointDescriptor baseOperation) {
    return operationBuilders.stream()
        .filter(x -> matchesOperationIdOrMethodAndPath(x, baseOperation))
        .findFirst();
  }

  private boolean matchesOperationIdOrMethodAndPath(OperationBuilder operationBuilder, BaseEndpointDescriptor baseOperation) {
    boolean matches = false;
    if (baseOperation.getOperationStringIdentifier() != null
        && baseOperation.getOperationStringIdentifier().equals(operationBuilder.getApiOperationId())) {
      matches = true;
    } else {
      String methodAndPath = format("%s-%s", operationBuilder.getHttpMethod().name(), operationBuilder.getPath());
      if (methodAndPath.equalsIgnoreCase(baseOperation.getOperationStringIdentifier())) {
        matches = true;
      }
    }
    return matches;
  }

  public Optional<TriggerBuilder> getTriggerBuildersByName(String triggerName) {
    return triggerBuilders.stream()
        .filter(x -> triggerName.equals(x.getName()))
        .findFirst();
  }

  public TriggerBuilder createTriggerAdapterBuilder(String triggerName, TriggerBuilder baseTriggerBuilder) {
    TriggerBuilder triggerBuilder = new TriggerBuilder(triggerName, baseTriggerBuilder);
    triggerBuilders.add(triggerBuilder);
    return triggerBuilder;
  }

  public OperationBuilder createOperationAdapterBuilder(String descriptorIdentifier, OperationBuilder baseOperationBuilder) {
    OperationBuilder operationBuilder = new OperationBuilder(descriptorIdentifier, baseOperationBuilder);
    operationBuilders.add(operationBuilder);
    return operationBuilder;
  }

  public OperationBuilder createNativeOperationAdapterBuilder(String descriptorIdentifier, String fqn) {
    OperationBuilder operationBuilder = new OperationBuilder(descriptorIdentifier, fqn);
    operationBuilders.add(operationBuilder);
    return operationBuilder;
  }

  public OperationBuilder getOrCreateOperationBuilder(String apiOperationId, String apiPath, String apiMethod,
                                                      String apiSummary) {
    requireNonNull(apiMethod, "API Method is required");
    requireNonNull(apiPath, "API Path is required");

    OperationBuilder operationBuilder =
        operationBuilders.stream()
            .filter(x -> x.getPath().equalsIgnoreCase(apiPath) && x.getMethod().equalsIgnoreCase(apiMethod))
            .findFirst().orElse(null);

    if (operationBuilder == null) {
      operationBuilder = new OperationBuilder(apiOperationId, apiPath, apiMethod, apiSummary);
      operationBuilders.add(operationBuilder);
    }

    return operationBuilder;
  }

  public PaginationBuilder getOrCreatePaginationBuilder(String paginationName) {
    requireNonNull(paginationName, "Pagination Name is required");

    PaginationBuilder paginationBuilder =
        paginationBuilders.stream()
            .filter(x -> x.getName().equals(paginationName))
            .findFirst().orElse(null);

    if (paginationBuilder == null) {
      paginationBuilder = new PaginationBuilder(paginationName);
      paginationBuilders.add(paginationBuilder);
    }

    return paginationBuilder;
  }

  public SecuritySchemeBuilder getOrCreateSecuritySchemeBuilder(String name,
                                                                ConnectorSecurityScheme.SecuritySchemeType type) {
    requireNonNull(name, "Security scheme name is required");

    SecuritySchemeBuilder securitySchemeBuilder =
        securitySchemeBuilders.stream()
            .filter(x -> x.getName().equals(name))
            .findFirst().orElse(null);

    if (securitySchemeBuilder == null) {
      requireNonNull(type, format("Security Scheme named %s does not exist and a type was not specified. This is a bug.", name));
      securitySchemeBuilder = new SecuritySchemeBuilder(name, type);
      securitySchemeBuilders.add(securitySchemeBuilder);
    }

    if (type != null && !securitySchemeBuilder.getType().equals(type)) {
      throw new IllegalStateException("Security scheme type does not match. This is a bug.");
    }

    return securitySchemeBuilder;
  }

  public List<TriggerBuilder> getTriggerBuilders() {
    return triggerBuilders;
  }

  public TriggerBuilder getOrCreateTriggerBuilder(String name) {
    requireNonNull(name, "Trigger name is required");

    TriggerBuilder triggerBuilder =
        triggerBuilders.stream()
            .filter(x -> x.getName().equals(name))
            .findFirst().orElse(null);

    if (triggerBuilder == null) {
      triggerBuilder = new TriggerBuilder(name);
      triggerBuilders.add(triggerBuilder);
    }

    return triggerBuilder;
  }

  public List<InterceptorBuilder> getInterceptorBuilders() {
    return interceptorBuilders;
  }

  public InterceptorBuilder createInterceptor() {

    InterceptorBuilder interceptorBuilder = new InterceptorBuilder();
    interceptorBuilders.add(interceptorBuilder);

    return interceptorBuilder;
  }

  public void setAllOperationsIgnored(Boolean ignoreOperations) {
    operationBuilders.forEach(x -> x.ignored(ignoreOperations));
    triggerBuilders.forEach(x -> x.ignored(ignoreOperations));
  }

  public ConnectorModel build() throws ModelGenerationException {
    requireNonNull(connectorName, "Connector name is required");

    TypeSchemaPool typeSchemaPool = new TypeSchemaPool();

    List<Pagination> paginations = paginationBuilders.stream().map(PaginationBuilder::build).collect(toList());

    OperationDisplayNameExpressionHandler operationDisplayNameExpressionHandler = buildOperationDisplayNameHandler();
    OperationIdentifierExpressionHandler operationIdentifierExpressionHandler = buildOperationIdentifierHandler();
    ParameterIdentifierExpressionHandler parameterIdentifierExpressionHandler = buildParameterIdentifierHandler();
    BodyIdentifierExpressionHandler bodyIdentifierExpressionHandler = buildBodyIdentifierHandler();

    List<ConnectorSecurityScheme> securitySchemes = buildSecuritySchemes(typeSchemaPool, parameterIdentifierExpressionHandler);

    List<ConnectorOperation> operations = operationBuilders.stream()
        .filter(operationBuilder -> StringUtils.isEmpty(operationBuilder.getFqn()))
        .map(x -> x.build(operationIdentifierExpressionHandler,
                          operationDisplayNameExpressionHandler,
                          parameterIdentifierExpressionHandler,
                          bodyIdentifierExpressionHandler,
                          typeSchemaPool,
                          paginations,
                          securitySchemes,
                          defaultInputMediaType,
                          defaultOutputMediaType,
                          buildQueryParamArrayFormat(),
                          skipOutputTypeValidation))
        .collect(toList());

    List<ConnectorOperation> nativeOperations = operationBuilders.stream()
        .filter(operationBuilder -> StringUtils.isNotBlank(operationBuilder.getFqn()))
        .map(x -> x.build())
        .collect(toList());

    operations.addAll(nativeOperations);

    List<Trigger> triggers = triggerBuilders.stream()
        .filter(x -> StringUtils.isEmpty(x.getFqn()))
        .filter(x -> !x.isIgnored())
        .map(x -> x.build(operations, typeSchemaPool)).collect(toList());

    List<Trigger> nativeTriggers = triggerBuilders.stream()
        .filter(x -> StringUtils.isNotBlank(x.getFqn()))
        .filter(x -> !x.isIgnored())
        .map(x -> x.buildNative()).collect(toList());

    triggers.addAll(nativeTriggers);
    return new ConnectorModel(buildConnectorName(),
                              connectorDescription,
                              buildMavenGroup(),
                              buildMavenArtifactId(),
                              buildMavenVersion(),
                              buildBasePackage(),
                              connectorCategory,
                              baseUriBuilder.build(supportedProtocols),
                              getSupportedProtocols(),
                              operations.stream().filter(x -> !x.isIgnored()).collect(toList()),
                              paginations,
                              buildExtensionXml(),
                              triggers,
                              interceptorBuilders.stream().map(x -> x.build()).collect(toList()),
                              connectorConfiguration);
  }

  private List<ConnectorSecurityScheme> buildSecuritySchemes(TypeSchemaPool typeSchemaPool,
                                                             ParameterIdentifierExpressionHandler parameterIdentifierExpressionHandler) {
    securitySchemeBuilders.add(new SecuritySchemeBuilder(UNSECURED.toString(), UNSECURED));

    return securitySchemeBuilders.stream()
        .filter(x -> x.isIgnored() == null || !x.isIgnored())
        .map(x -> x.build(typeSchemaPool, testConnectionConfig, parameterIdentifierExpressionHandler))
        .collect(toList());
  }

  private String buildConnectorName() {
    String builtName = connectorName.trim();

    if (builtName.toLowerCase().endsWith("api")) {
      builtName = builtName.substring(0, builtName.lastIndexOf("api"));
    }
    if (builtName.toLowerCase().endsWith("connector")) {
      return builtName;
    }
    return builtName + " Connector";
  }

  private String buildMavenGroup() {
    return removeMavenGroupUnwantedCharacters(mavenGroup);
  }

  private String buildMavenArtifactId() {
    return removeMavenArtifactUnwantedCharacters(mavenArtifactId);
  }

  private String buildMavenVersion() {
    return removeMavenVersionUnwantedCharacters(mavenVersion);
  }

  private List<Protocol> getSupportedProtocols() {
    return (supportedProtocols != null && !supportedProtocols.isEmpty()) ? supportedProtocols : asList(DEFAULT_PROTOCOLS);
  }

  private String buildBasePackage() {
    return isNotBlank(baseJavaPackage) ? baseJavaPackage : ConnectorPackage.buildBasePackage(connectorName);
  }

  private String buildExtensionXml() {
    return isNotBlank(extensionXml) ? extensionXml : getXmlName(connectorName);
  }

  private QueryParamArrayFormat buildQueryParamArrayFormat() {
    return defaultIfNull(queryParamArrayFormat, MULTIMAP);
  }

  private OperationIdentifierExpressionHandler buildOperationIdentifierHandler() {
    try {
      return isNotBlank(operationIdentifierExpression)
          ? new OperationIdentifierExpressionHandler(operationIdentifierExpression)
          : new OperationIdentifierExpressionHandler();

    } catch (CompilationException e) {
      throw new IllegalArgumentException("The script for the 'operationIdentifier' failed to compile. This is a bug.", e);
    }
  }

  private OperationDisplayNameExpressionHandler buildOperationDisplayNameHandler() {
    try {
      return isNotBlank(operationDisplayNameExpression)
          ? new OperationDisplayNameExpressionHandler(operationDisplayNameExpression)
          : new OperationDisplayNameExpressionHandler();
    } catch (CompilationException e) {
      throw new IllegalArgumentException("The script for the 'operationDisplayName' failed to compile. This is a bug.", e);
    }
  }

  private ParameterIdentifierExpressionHandler buildParameterIdentifierHandler() {
    try {
      return isNotBlank(parameterIdentifierExpression)
          ? new ParameterIdentifierExpressionHandler(parameterIdentifierExpression)
          : new ParameterIdentifierExpressionHandler();

    } catch (CompilationException e) {
      throw new IllegalArgumentException("The script for the 'parameterIdentifier' failed to compile. This is a bug.", e);
    }
  }

  private BodyIdentifierExpressionHandler buildBodyIdentifierHandler() {
    try {
      return isNotBlank(bodyIdentifierExpression)
          ? new BodyIdentifierExpressionHandler(bodyIdentifierExpression)
          : new BodyIdentifierExpressionHandler();

    } catch (CompilationException e) {
      throw new IllegalArgumentException("The script for the 'bodyIdentifier' failed to compile. This is a bug.", e);
    }
  }
}
