/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder;

import static com.google.common.collect.Lists.newArrayList;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.ConnectorSecurityScheme.SecuritySchemeType.CUSTOM_AUTHENTICATION;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.ConnectorSecurityScheme.SecuritySchemeType.UNSECURED;
import static java.util.Objects.requireNonNull;
import static java.util.stream.Collectors.toList;
import static javax.ws.rs.core.MediaType.APPLICATION_JSON_TYPE;
import static javax.ws.rs.core.MediaType.APPLICATION_XML_TYPE;
import static javax.ws.rs.core.MediaType.MULTIPART_FORM_DATA_TYPE;
import static org.apache.commons.lang3.ObjectUtils.defaultIfNull;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.operation.ConnectorNativeOperation;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.operation.ConnectorOperation;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.HTTPMethod;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.QueryParamArrayFormat;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.TypeSchemaPool;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.body.Body;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.resolver.sampledata.SampleDataBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dw.BodyIdentifierExpressionHandler;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dw.OperationDisplayNameExpressionHandler;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dw.OperationIdentifierExpressionHandler;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dw.ParameterIdentifierExpressionHandler;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.pagination.Pagination;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.AuxiliarParameter;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.Parameter;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterBinding;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.ConnectorSecurityScheme;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.CustomAuthenticationScheme;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.TypeDefinition;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.util.ComparisonUtil;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import javax.ws.rs.core.MediaType;

public class OperationBuilder {

  private String fqn;
  private final String apiOperationId;
  private final String path;
  private final String method;
  private List<String> baseUris = new ArrayList<>();
  private final String apiSummary;

  private String descriptorIdentifier;
  private String displayName;
  private String description;
  private String alias;
  private final List<ParameterBuilder> uriParameterBuilders = new ArrayList<>();
  private final List<ParameterBuilder> queryParameterBuilders = new ArrayList<>();
  private final List<ParameterBuilder> headerBuilders = new ArrayList<>();
  private final BodyBuilder bodyBuilder = new BodyBuilder();
  private final Map<MediaType, TypeDefinitionBuilder> inputMetadataBuilders = new HashMap<>();
  private final Map<MediaType, TypeDefinitionBuilder> outputMetadataBuilders = new HashMap<>();
  private List<String> securitySchemeIds = new ArrayList<>();
  private String alternativeBaseUri;
  private String pagination;
  private Boolean skipOutputTypeValidation;
  private Boolean isVoidOperation;
  private QueryParamArrayFormat queryParamArrayFormat;
  private Boolean ignored;
  private Boolean privateOperation;
  private SampleDataBuilder sampleDataBuilder = new SampleDataBuilder();
  private MediaType defaultInputMediaType;
  private MediaType defaultOutputMediaType;
  private String forcedOutputTypeSchema;
  private String forcedInputTypeSchema;
  private String muleOutputResolver;
  private boolean isAdapter = false;
  private boolean isSidecar = false;
  private boolean isRefined = false;
  private final List<AuxiliarParameterBuilder> parameters = new ArrayList<>();
  private final List<AuxiliarParameterBindingBuilder> requestBindings = new ArrayList<>();
  private final List<AuxiliarParameterBindingBuilder> responseBindings = new ArrayList<>();

  OperationBuilder(String descriptorIdentifier, String fqn) {
    this.descriptorIdentifier = descriptorIdentifier;
    this.fqn = fqn;
    this.apiOperationId = null;
    this.path = null;
    this.method = null;
    this.apiSummary = null;
  }

  OperationBuilder(String descriptorIdentifier, OperationBuilder baseOperationBuilder) {
    this.descriptorIdentifier = descriptorIdentifier;
    this.method = baseOperationBuilder.method;
    this.path = baseOperationBuilder.path;
    this.baseUris = baseOperationBuilder.baseUris;
    this.apiSummary = baseOperationBuilder.apiSummary;
    this.apiOperationId = null;

    this.displayName = baseOperationBuilder.displayName;
    this.description = baseOperationBuilder.description;
    this.uriParameterBuilders.addAll(baseOperationBuilder.uriParameterBuilders);
    this.queryParameterBuilders.addAll(baseOperationBuilder.queryParameterBuilders);
    this.headerBuilders.addAll(baseOperationBuilder.headerBuilders);
    this.inputMetadataBuilders.putAll(baseOperationBuilder.inputMetadataBuilders);
    this.outputMetadataBuilders.putAll(baseOperationBuilder.outputMetadataBuilders);
    securitySchemeIds(baseOperationBuilder.securitySchemeIds);
    this.alternativeBaseUri = baseOperationBuilder.alternativeBaseUri;
    this.pagination = baseOperationBuilder.pagination;
    this.skipOutputTypeValidation = baseOperationBuilder.skipOutputTypeValidation;
    this.isVoidOperation = baseOperationBuilder.isVoidOperation;
    this.queryParamArrayFormat = baseOperationBuilder.queryParamArrayFormat;
    this.ignored = baseOperationBuilder.ignored;
    this.privateOperation = baseOperationBuilder.privateOperation;
    this.alias = baseOperationBuilder.alias;

    this.sampleDataBuilder = new SampleDataBuilder(baseOperationBuilder.sampleDataBuilder);
    this.defaultInputMediaType = baseOperationBuilder.defaultInputMediaType;
    this.defaultOutputMediaType = baseOperationBuilder.defaultOutputMediaType;
    this.forcedOutputTypeSchema = baseOperationBuilder.forcedOutputTypeSchema;
    this.forcedInputTypeSchema = baseOperationBuilder.forcedInputTypeSchema;
    this.isAdapter = baseOperationBuilder.isAdapter;
  }

  OperationBuilder(String apiOperationId, String path, String method, String apiSummary) {
    requireNonNull(path);
    requireNonNull(method);

    this.method = method;
    this.path = path;
    this.apiOperationId = apiOperationId;
    this.apiSummary = apiSummary;
    this.descriptorIdentifier = null;
  }

  public OperationBuilder displayName(String displayName) {
    this.displayName = defaultIfNull(displayName, this.displayName);
    return this;
  }

  public OperationBuilder descriptorIdentifier(String descriptorIdentifier) {
    this.descriptorIdentifier = defaultIfNull(descriptorIdentifier, this.descriptorIdentifier);
    return this;
  }

  public OperationBuilder description(String description) {
    this.description = defaultIfNull(description, this.description);
    return this;
  }

  public OperationBuilder alias(String alias) {
    this.alias = defaultIfNull(alias, this.alias);
    return this;
  }

  public String getAlias() {
    return alias;
  }

  public OperationBuilder securitySchemes(List<String> securitySchemeIds) {
    this.securitySchemeIds = defaultIfNull(securitySchemeIds, this.securitySchemeIds);
    return this;
  }

  public OperationBuilder alternativeBaseUri(String alternativeBaseUri) {
    this.alternativeBaseUri = defaultIfNull(alternativeBaseUri, this.alternativeBaseUri);
    return this;
  }

  public boolean hasPagination() {
    return isNotBlank(pagination);
  }

  public OperationBuilder pagination(String pagination) {
    this.pagination = defaultIfNull(pagination, this.pagination);
    return this;
  }

  public Boolean getSkipOutputTypeValidation() {
    return this.skipOutputTypeValidation;
  }

  public OperationBuilder skipOutputTypeValidation(Boolean skipOutputTypeValidation) {
    this.skipOutputTypeValidation = defaultIfNull(skipOutputTypeValidation, this.skipOutputTypeValidation);
    return this;
  }

  public Boolean getIsVoidOperation() {
    return this.isVoidOperation;
  }

  public OperationBuilder voidOperation(Boolean isVoidOperation) {
    this.isVoidOperation = defaultIfNull(isVoidOperation, this.isVoidOperation);
    return this;
  }

  public Boolean isIgnored() {
    return ignored;
  }

  public OperationBuilder ignored(Boolean ignored) {
    this.ignored = defaultIfNull(ignored, this.ignored);
    return this;
  }

  public Boolean isPrivateOperation() {
    return privateOperation;
  }

  public OperationBuilder privateOperation(Boolean privateOperation) {
    this.privateOperation = defaultIfNull(privateOperation, this.privateOperation);
    return this;
  }

  public OperationBuilder queryParamArrayFormat(QueryParamArrayFormat queryParamArrayFormat) {
    this.queryParamArrayFormat = defaultIfNull(queryParamArrayFormat, this.queryParamArrayFormat);
    return this;
  }

  public MediaType getDefaultInputMediaType() {
    return defaultInputMediaType;
  }

  public OperationBuilder setDefaultInputMediaType(MediaType defaultInputMediaType) {
    this.defaultInputMediaType = defaultIfNull(defaultInputMediaType, this.defaultInputMediaType);
    return this;
  }

  public MediaType getDefaultOutputMediaType() {
    return defaultOutputMediaType;
  }

  public OperationBuilder setDefaultOutputMediaType(MediaType defaultOutputMediaType) {
    this.defaultOutputMediaType = defaultIfNull(defaultOutputMediaType, this.defaultOutputMediaType);
    return this;
  }

  public OperationBuilder parameters(List<AuxiliarParameterBuilder> parameters) {
    this.parameters.clear();
    this.parameters.addAll(parameters);
    return this;
  }

  public OperationBuilder requestBindings(List<AuxiliarParameterBindingBuilder> requestBindings) {
    this.requestBindings.clear();
    if (requestBindings != null) {
      this.requestBindings.addAll(requestBindings);
    }
    return this;
  }

  public OperationBuilder responseBindings(List<AuxiliarParameterBindingBuilder> responseBindings) {
    this.responseBindings.clear();
    if (responseBindings != null) {
      this.responseBindings.addAll(responseBindings);
    }
    return this;
  }

  public OperationBuilder baseUris(List<String> baseUris) {
    this.baseUris.clear();
    if (baseUris != null) {
      this.baseUris.addAll(baseUris);
    }
    return this;
  }

  public OperationBuilder uriParameters(List<ParameterBuilder> uriParameters) {
    this.uriParameterBuilders.clear();
    if (uriParameters != null) {
      this.uriParameterBuilders.addAll(uriParameters);
    }
    return this;
  }

  public OperationBuilder queryParameters(List<ParameterBuilder> queryParameterBuilders) {
    this.queryParameterBuilders.clear();
    if (queryParameterBuilders != null) {
      this.queryParameterBuilders.addAll(queryParameterBuilders);
    }
    return this;
  }

  public OperationBuilder headers(List<ParameterBuilder> headerBuilders) {
    this.headerBuilders.clear();
    if (headerBuilders != null) {
      this.headerBuilders.addAll(headerBuilders);
    }
    return this;
  }

  public OperationBuilder securitySchemeIds(List<String> securitySchemeIds) {
    if (this.securitySchemeIds == null) {
      this.securitySchemeIds = new ArrayList<>();
    } else {
      this.securitySchemeIds.clear();
    }
    this.securitySchemeIds.addAll(securitySchemeIds);
    return this;
  }

  public OperationBuilder adapter() {
    this.isAdapter = true;
    return this;
  }

  public OperationBuilder sidecar() {
    this.isSidecar = true;
    return this;
  }

  public OperationBuilder refined(Boolean refined) {
    this.isRefined = defaultIfNull(refined, this.isRefined);
    return this;
  }

  public String getDescriptorIdentifier() {
    return descriptorIdentifier;
  }

  public String getPath() {
    return path;
  }

  public String getApiOperationId() {
    return apiOperationId;
  }

  public String getMethod() {
    return method;
  }

  public boolean isAdapter() {
    return isAdapter;
  }

  public boolean hasForcedOutputTypeSchema() {
    return forcedOutputTypeSchema != null;
  }

  public OperationBuilder forceOutputTypeSchema(String forcedOutputTypeSchema) {
    this.forcedOutputTypeSchema = defaultIfNull(forcedOutputTypeSchema, this.forcedOutputTypeSchema);
    return this;
  }

  public boolean hasForcedInputTypeSchema() {
    return forcedInputTypeSchema != null;
  }

  public String getFqn() {
    return fqn;
  }

  public OperationBuilder forceInputTypeSchema(String forcedInputTypeSchema) {
    this.forcedInputTypeSchema = defaultIfNull(forcedInputTypeSchema, this.forcedInputTypeSchema);;
    return this;
  }

  public OperationBuilder muleOutputResolver(String muleOutputResolver) {
    this.muleOutputResolver = defaultIfNull(muleOutputResolver, this.muleOutputResolver);
    return this;
  }

  public List<TypeDefinitionBuilder> getMultipartInputMetadataBuilders() {
    return inputMetadataBuilders
        .values().stream()
        .filter(TypeDefinitionBuilder::isMultipart)
        .collect(toList());
  }

  public Map<MediaType, TypeDefinitionBuilder> getInputMetadataBuilders() {
    return inputMetadataBuilders;
  }

  public TypeDefinitionBuilder getOrCreateInputMetadataBuilder(MediaType mediaType) {
    return getOrCreateTypeDefinitionBuilder(inputMetadataBuilders, mediaType);
  }

  public Map<MediaType, TypeDefinitionBuilder> getOutputMetadataBuilders() {
    return outputMetadataBuilders;
  }

  public TypeDefinitionBuilder getOrCreateOutputMetadataBuilder(MediaType mediaType) {
    return getOrCreateTypeDefinitionBuilder(outputMetadataBuilders, mediaType);
  }

  private TypeDefinitionBuilder getOrCreateTypeDefinitionBuilder(Map<MediaType, TypeDefinitionBuilder> map,
                                                                 MediaType mediaType) {
    TypeDefinitionBuilder typeDefinitionBuilder = map.get(mediaType);
    if (typeDefinitionBuilder == null) {
      typeDefinitionBuilder = new TypeDefinitionBuilder();
      map.put(mediaType, typeDefinitionBuilder);
    }
    return typeDefinitionBuilder;
  }

  public ParameterBuilder getOrCreateParameterBuilder(ParameterType parameterType, String externalName,
                                                      String parameterIdentifier) {
    // TODO code smell quite similar to
    // com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.SecuritySchemeBuilder.getOrCreateParameterBuilder
    List<ParameterBuilder> parameterBuilders = getParameterBuilders(parameterType);

    ParameterBuilder paramBuilder = parameterBuilders.stream()
        .filter(x -> ComparisonUtil.externalNameParamsComparison(x.getExternalName(), externalName, parameterType))
        .findFirst().orElse(null);

    if (paramBuilder == null) {
      paramBuilder = new ParameterBuilder(parameterType, externalName, parameterIdentifier);
      parameterBuilders.add(paramBuilder);
    }

    return paramBuilder;
  }

  public List<String> getBaseUris() {
    return baseUris;
  }

  private List<ParameterBuilder> getParameterBuilders(ParameterType parameterType) {
    switch (parameterType) {
      case URI:
        return uriParameterBuilders;
      case QUERY:
        return queryParameterBuilders;
      case HEADER:
        return headerBuilders;
      default:
        throw new IllegalArgumentException("Parameter Type not supported. This is a bug.");
    }
  }

  public SampleDataBuilder getSampleDataBuilder() {
    return sampleDataBuilder;
  }

  ConnectorOperation build() {
    return new ConnectorNativeOperation(descriptorIdentifier,
                                        privateOperation != null && privateOperation,
                                        this.fqn);
  }

  ConnectorOperation build(OperationIdentifierExpressionHandler identifierHandler,
                           OperationDisplayNameExpressionHandler displayNameHandler,
                           ParameterIdentifierExpressionHandler parameterIdentifierExpressionHandler,
                           BodyIdentifierExpressionHandler bodyIdentifierExpressionHandler,
                           TypeSchemaPool typeSchemaPool,
                           List<Pagination> paginations,
                           List<ConnectorSecurityScheme> securitySchemes,
                           MediaType globalDefaultInputMediaType,
                           MediaType globalDefaultOutputMediaType,
                           QueryParamArrayFormat globalQueryParamArrayFormat,
                           Boolean globalSkipOutputTypeValidation) {

    Pagination pagination = this.pagination != null ? paginations.stream()
        .filter(x -> x.getName().equals(this.pagination))
        .findFirst()
        .orElseThrow(() -> new IllegalArgumentException("Pagination not found. This is a bug."))
        : null;

    addPartIdentifiersForMultipartScenarios(bodyIdentifierExpressionHandler, apiOperationId, method, path);

    ConnectorOperation operation =
        new ConnectorOperation(
                               descriptorIdentifier != null ? descriptorIdentifier
                                   : identifierHandler.evaluate(apiOperationId, method, path),
                               displayName != null ? displayName
                                   : displayNameHandler.evaluate(apiOperationId, method, path, apiSummary),
                               description,
                               path,
                               alias,
                               baseUris,
                               HTTPMethod.fromString(method),
                               buildParameters(uriParameterBuilders, typeSchemaPool,
                                               parameterIdentifierExpressionHandler),
                               buildParameters(queryParameterBuilders, typeSchemaPool,
                                               parameterIdentifierExpressionHandler),
                               buildHeaders(headerBuilders, typeSchemaPool,
                                            parameterIdentifierExpressionHandler),
                               buildAuxiliarParameters(parameters, typeSchemaPool),
                               buildParameterBindings(requestBindings),
                               buildParameterBindings(responseBindings),
                               buildBody(bodyIdentifierExpressionHandler),
                               buildIOType(inputMetadataBuilders,
                                           forcedInputTypeSchema,
                                           defaultIfNull(defaultInputMediaType, globalDefaultInputMediaType),
                                           typeSchemaPool),
                               buildIOType(outputMetadataBuilders,
                                           forcedOutputTypeSchema,
                                           defaultIfNull(defaultOutputMediaType, globalDefaultOutputMediaType),
                                           typeSchemaPool),
                               muleOutputResolver,
                               buildSecuritySchemes(securitySchemes),
                               alternativeBaseUri,
                               pagination,
                               defaultIfNull(skipOutputTypeValidation, globalSkipOutputTypeValidation),
                               isVoidOperation,

                               buildQueryParamArrayFormat(globalQueryParamArrayFormat),
                               ignored != null && ignored, privateOperation != null && privateOperation,
                               isAdapter, isSidecar, this.fqn, isRefined);

    operation.setSampleData(sampleDataBuilder.build(operation));

    return operation;
  }

  private void addPartIdentifiersForMultipartScenarios(BodyIdentifierExpressionHandler bodyIdentifierExpressionHandler,
                                                       String apiOperationId, String method, String path) {

    inputMetadataBuilders.entrySet().stream()
        .filter(entry -> {
          // filtering elements that are multipart
          return MULTIPART_FORM_DATA_TYPE.isCompatible(entry.getKey())
              // and they have also parts to work with
              && entry.getValue().getParts() != null;
        }).forEach(entry -> {
          entry.getValue().getParts()
              .stream().filter(parameterBuilder -> {
                // we will only evaluate the bodyIdentifier script if there's no partIdentifier in place
                return parameterBuilder.getParameterIdentifier() == null;
              }).forEach(parameterBuilder -> {
                String partName = parameterBuilder.getExternalName();
                String partNameIdentifier = bodyIdentifierExpressionHandler.evaluate(apiOperationId, method, path, partName);
                parameterBuilder.parameterIdentifier(partNameIdentifier);
              });
        });
  }

  private Body buildBody(BodyIdentifierExpressionHandler bodyIdentifierExpressionHandler) {
    return bodyBuilder.buildBody(bodyIdentifierExpressionHandler, apiOperationId, method, path);
  }

  private List<ParameterBinding> buildParameterBindings(List<AuxiliarParameterBindingBuilder> bindings) {
    return bindings.stream().map(AuxiliarParameterBindingBuilder::build).collect(toList());
  }

  public BodyBuilder getBodyBuilder() {
    return bodyBuilder;
  }

  private List<ConnectorSecurityScheme> buildSecuritySchemes(List<ConnectorSecurityScheme> securitySchemes) {

    if (securitySchemeIds == null || securitySchemeIds.isEmpty()) {
      Optional<ConnectorSecurityScheme> customSecurityScheme = securitySchemes.stream()
          .filter(x -> x.getSchemeType().equals(CUSTOM_AUTHENTICATION))
          .filter(x -> ((CustomAuthenticationScheme) x).getFqn().isPresent())
          .findAny();

      return newArrayList(customSecurityScheme.orElseGet(() -> securitySchemes.stream()
          .filter(x -> x.getName().equalsIgnoreCase(UNSECURED.toString()))
          .findAny()
          .orElseThrow(() -> new IllegalStateException("There should be at least an unsecured security scheme. This is a bug."))));
    }

    return securitySchemes.stream()
        .filter(x -> securitySchemeIds.contains(x.getName()) || x.getSchemeType().equals(CUSTOM_AUTHENTICATION))
        .collect(toList());
  }

  private QueryParamArrayFormat buildQueryParamArrayFormat(QueryParamArrayFormat globalQueryParamArrayFormat) {
    return defaultIfNull(this.queryParamArrayFormat, globalQueryParamArrayFormat);
  }

  private TypeDefinition buildIOType(Map<MediaType, TypeDefinitionBuilder> builders,
                                     String forcedTypeSchema,
                                     MediaType defaultMediaType,
                                     TypeSchemaPool typeSchemaPool) {

    TypeDefinitionBuilder typeBuilder =
        getTypeDefinitionBuilderForMediaTypeOrDefault(builders, defaultMediaType);

    if (typeBuilder != null) {
      return typeBuilder.build(typeSchemaPool, forcedTypeSchema);
    }

    if (forcedTypeSchema != null) {
      return new TypeDefinitionBuilder().object().build(typeSchemaPool, forcedTypeSchema);
    }

    return null;
  }

  private List<Parameter> buildHeaders(List<ParameterBuilder> headerBuilders, TypeSchemaPool typeSchemaPool,
                                       ParameterIdentifierExpressionHandler parameterIdentifierExpressionHandler) {
    // Accept header will be inferred from the response media type, so we remove it from the operation parameters.
    return buildParameters(
                           headerBuilders.stream().filter(x -> !x.getExternalName().equalsIgnoreCase("accept")).collect(toList()),
                           typeSchemaPool, parameterIdentifierExpressionHandler);
  }

  private List<Parameter> buildParameters(List<ParameterBuilder> parameterBuilders, TypeSchemaPool typeSchemaPool,
                                          ParameterIdentifierExpressionHandler parameterIdentifierExpressionHandler) {

    return parameterBuilders.stream()
        .filter(x -> x.isIgnored() == null || !x.isIgnored())
        .map(x -> x.buildParameter(typeSchemaPool, parameterIdentifierExpressionHandler))
        .collect(toList());
  }

  private List<AuxiliarParameter> buildAuxiliarParameters(List<AuxiliarParameterBuilder> parameterBuilders,
                                                          TypeSchemaPool typeSchemaPool) {

    return parameterBuilders.stream()
        .map(x -> x.build(typeSchemaPool))
        .collect(toList());
  }

  private TypeDefinitionBuilder getTypeDefinitionBuilderForMediaTypeOrDefault(Map<MediaType, TypeDefinitionBuilder> builders,
                                                                              MediaType mediaType) {
    if (builders == null || builders.isEmpty()) {
      return null;
    }

    if (mediaType == null) {
      return getDefaultTypeDefinitionBuilder(builders);
    }

    TypeDefinitionBuilder typeDefinitionBuilder = builders.get(mediaType);

    if (typeDefinitionBuilder != null) {
      return typeDefinitionBuilder;
    }

    return getDefaultTypeDefinitionBuilder(builders);
  }

  private TypeDefinitionBuilder getDefaultTypeDefinitionBuilder(Map<MediaType, TypeDefinitionBuilder> builders) {
    if (builders.get(APPLICATION_JSON_TYPE) != null) {
      return builders.get(APPLICATION_JSON_TYPE);
    }

    else if (builders.get(APPLICATION_XML_TYPE) != null) {
      return builders.get(APPLICATION_XML_TYPE);
    }

    return builders.values().stream().findFirst().orElse(null);
  }

  public String getDisplayName() {
    return displayName;
  }

  public String getDescription() {
    return description;
  }

  public List<ParameterBuilder> getUriParameters() {
    return Collections.unmodifiableList(uriParameterBuilders);
  }

  public List<ParameterBuilder> getQueryParameters() {
    return Collections.unmodifiableList(queryParameterBuilders);
  }

  public List<ParameterBuilder> getHeaders() {
    return Collections.unmodifiableList(headerBuilders);
  }

  public Map<MediaType, TypeDefinitionBuilder> getInputMetadata() {
    return inputMetadataBuilders;
  }

  public HTTPMethod getHttpMethod() {
    return HTTPMethod.fromString(method);
  }

}
