/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.PART;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.util.NamingUtil.isFriendlyName;
import static java.util.Objects.requireNonNull;
import static org.apache.commons.lang3.ObjectUtils.defaultIfNull;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.TypeSchemaPool;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.resolver.valueprovider.ValueProviderBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dw.ParameterIdentifierExpressionHandler;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.Parameter;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.PartParameter;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.PrimitiveTypeDefinition;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.TypeDefinition;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.util.NamingUtil;

import java.util.Optional;

public class ParameterBuilder {

  private final String externalName;
  private final ParameterType parameterType;

  private String parameterIdentifier;
  private String displayName;
  private String muleAlias;
  private String muleMetadataKeyId;
  private final TypeDefinitionBuilder typeDefinitionBuilder = new TypeDefinitionBuilder();
  private String description;
  private Boolean required;
  private String defaultValue;
  private Boolean password;
  private Boolean filePart;
  private final ValueProviderBuilder valueProviderExpressionBuilder = new ValueProviderBuilder();
  private Boolean ignored;

  public ParameterBuilder(ParameterType parameterType, String externalName, String parameterIdentifier) {
    requireNonNull(parameterType);
    requireNonNull(externalName);

    this.parameterType = parameterType;
    this.externalName = externalName;
    this.parameterIdentifier = parameterIdentifier;
  }

  public String getExternalName() {
    return externalName;
  }

  public String getParameterIdentifier() {
    return parameterIdentifier;
  }

  public ParameterBuilder parameterIdentifier(String parameterIdentifier) {
    this.parameterIdentifier = defaultIfNull(parameterIdentifier, this.parameterIdentifier);
    return this;
  }

  public ParameterType getParameterType() {
    return parameterType;
  }

  public Boolean isIgnored() {
    return ignored;
  }

  public Boolean isRequired() {
    return required;
  }

  public ParameterBuilder displayName(String displayName) {
    this.displayName = defaultIfNull(displayName, this.displayName);
    return this;
  }

  public TypeDefinitionBuilder getTypeDefinitionBuilder() {
    return this.typeDefinitionBuilder;
  }

  public ParameterBuilder description(String description) {
    this.description = defaultIfNull(description, this.description);
    return this;
  }

  public String getMuleAlias() {
    return this.muleAlias;
  }

  public String getMuleMetadataKeyId() {
    return this.muleMetadataKeyId;
  }

  public ParameterBuilder muleAlias(String muleAlias) {
    this.muleAlias = defaultIfNull(muleAlias, this.muleAlias);
    return this;
  }

  public ParameterBuilder muleMetadataKeyId(String muleMetadataKeyId) {
    this.muleMetadataKeyId = defaultIfNull(muleMetadataKeyId, this.muleMetadataKeyId);
    return this;
  }

  public ParameterBuilder required(Boolean required) {
    this.required = defaultIfNull(required, this.required);
    return this;
  }

  public ParameterBuilder defaultValue(String defaultValue) {
    this.defaultValue = defaultIfNull(defaultValue, this.defaultValue);
    return this;
  }

  public ParameterBuilder password(Boolean password) {
    this.password = defaultIfNull(password, this.password);
    return this;
  }

  public ParameterBuilder filePart(Boolean isFilePart) {
    this.filePart = defaultIfNull(isFilePart, this.filePart);
    return this;
  }

  public ParameterBuilder partType(Boolean isFilePart) {
    this.filePart = defaultIfNull(isFilePart, this.filePart);
    return this;
  }

  public ParameterBuilder ignored(Boolean ignored) {
    this.ignored = defaultIfNull(ignored, this.ignored);
    return this;
  }

  public ValueProviderBuilder getValueProviderBuilder() {
    return this.valueProviderExpressionBuilder;
  }

  public Parameter buildParameter(TypeSchemaPool typeSchemaPool,
                                  Optional<ParameterIdentifierExpressionHandler> parameterIdentifierExpressionHandlerOptional) {
    Parameter parameter = new Parameter(buildDisplayName(),
                                        externalName,
                                        buildParameterIdentifier(parameterIdentifierExpressionHandlerOptional),
                                        parameterType,
                                        buildParameterTypeDefinition(typeSchemaPool),
                                        muleAlias,
                                        description,
                                        required == null || required,
                                        defaultValue,
                                        password != null && password,
                                        valueProviderExpressionBuilder.build(),
                                        muleMetadataKeyId,
                                        null,
                                        null);

    return parameter;
  }

  public Parameter buildParameter(TypeSchemaPool typeSchemaPool,
                                  ParameterIdentifierExpressionHandler parameterIdentifierExpressionHandler) {
    return buildParameter(typeSchemaPool, Optional.ofNullable(parameterIdentifierExpressionHandler));
  }

  private String buildParameterIdentifier(Optional<ParameterIdentifierExpressionHandler> parameterIdentifierExpressionHandlerOptional) {

    String result = parameterIdentifier;
    if (parameterIdentifier == null && parameterIdentifierExpressionHandlerOptional.isPresent()) {
      ParameterIdentifierExpressionHandler.TypeParam typeParam =
          ParameterIdentifierExpressionHandler.TypeParam.valueOf(parameterType.toString());
      result = parameterIdentifierExpressionHandlerOptional.get().evaluate(externalName, typeParam);
    }
    return result;
  }

  public PartParameter buildPartParameter(TypeSchemaPool typeSchemaPool) {
    if (!parameterType.equals(PART)) {
      throw new IllegalArgumentException("Parameter type not supported. This is a bug.");
    }

    requireNonNull(typeDefinitionBuilder);

    return new PartParameter(parameterIdentifier,
                             buildDisplayName(),
                             externalName,
                             buildParameterTypeDefinition(typeSchemaPool),
                             description,
                             filePart != null && filePart);
  }

  private String buildDisplayName() {
    if (displayName != null) {
      return displayName;
    }

    if (!isFriendlyName(externalName)) {
      // TODO RSDK-368: Display name generation by comprehension
      return NamingUtil.makeNameFriendly(externalName);
    }

    return externalName;
  }

  private TypeDefinition buildParameterTypeDefinition(TypeSchemaPool typeSchemaPool) {
    if (typeDefinitionBuilder.canBuild()) {
      return typeDefinitionBuilder.build(typeSchemaPool, null);
    }

    return new TypeDefinitionBuilder()
        .primitive(PrimitiveTypeDefinition.PrimitiveType.STRING)
        .build(typeSchemaPool, null);
  }
}
