/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dw;

import static java.lang.String.format;

import static java.util.Optional.empty;

import org.mule.weave.v2.model.ServiceManager;
import org.mule.weave.v2.parser.phase.CompilationException;
import org.mule.weave.v2.runtime.DataWeaveResult;
import org.mule.weave.v2.runtime.DataWeaveScript;
import org.mule.weave.v2.runtime.DataWeaveScriptingEngine;
import org.mule.weave.v2.runtime.ScriptingBindings;

import java.net.URL;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

public class ExpressionHandlerUtils {

  private static final String APPLICATION_JAVA = "application/java";

  /**
   * @param weaveScript parameterized weave script which might have the extra '#['/']' we need to remove
   * @throws CompilationException if the user consumes bindings that are not present, taken from {@code #implicitInputs}
   */
  public static DataWeaveScript compileDataWeaveScript(String weaveScript, String[] implicitInputs)
      throws CompilationException {
    DataWeaveScriptingEngine scriptingEngine = new DataWeaveScriptingEngine();
    return scriptingEngine.compile(curateScript(weaveScript), implicitInputs);
  }

  static DataWeaveScript compileDataWeaveScript(Class clazz, String weaveScriptFile,
                                                String[] implicitInputs) {
    DataWeaveScriptingEngine scriptingEngine = new DataWeaveScriptingEngine();
    URL resourceWeave = getResourceWeave(clazz, weaveScriptFile);
    return scriptingEngine.compile(resourceWeave, implicitInputs);
  }

  private static URL getResourceWeave(Class clazz, String weaveScriptFile) {
    URL resource = clazz.getResource(weaveScriptFile);
    if (resource == null) {
      throw new RuntimeException(format("Could not find DWL file '%s' in the resources. This is a bug.", weaveScriptFile));
    }
    return resource;
  }

  public static String evaluate(DataWeaveScript script, Map<String, String> map, String contextInvoker) {
    ScriptingBindings bindings = getBindings(map);
    // we parameterize the outputMimeType to avoid asking each user to feed it in their scripts, we will default to
    // application/java, specially as we are going to consume the result as a string afterwards
    DataWeaveResult dataWeaveResult = script.write(bindings, ServiceManager.apply(), APPLICATION_JAVA);
    if (!(dataWeaveResult.getContent() instanceof String)) {
      // if content is not a string, we cannot extract the content
      throw new IllegalArgumentException(format("Expression for [%s] must be a literal value, bindings used [%s]",
                                                contextInvoker,
                                                map.entrySet()
                                                    .stream()
                                                    .sorted(Map.Entry.comparingByValue())
                                                    .map(entry -> entry.getKey() + "=" + entry.getValue())
                                                    .collect(Collectors.joining(", "))));
    }
    return dataWeaveResult.getContentAsString();
  }

  public static Optional<String> getDeclaredOutputMimeType(String script) {
    try {
      DataWeaveScript dataWeaveScript = compileDataWeaveScript(curateScript(script), new String[0]);
      return Optional.ofNullable(dataWeaveScript.declaredOutputMimeType().get());
    } catch (CompilationException e) {
      return empty();
    }
  }

  private static ScriptingBindings getBindings(Map<String, String> map) {
    ScriptingBindings scriptingBindings = new ScriptingBindings();
    map.forEach((k, v) -> scriptingBindings.addBinding(k, v, APPLICATION_JAVA, new HashMap<>()));
    return scriptingBindings;
  }

  /**
   * Only needed when the script comes from a user
   *
   * @param script from the user
   * @return a trimmed script without the '#['/']'
   */
  private static String curateScript(String script) {
    String trimmedScript = script.trim();
    if (trimmedScript.startsWith("#[") && trimmedScript.endsWith("]")) {
      trimmedScript = trimmedScript.substring(2, trimmedScript.length() - 1);
    }
    return trimmedScript;
  }

}
