/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.connectormodel.loader.descriptor;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.loader.descriptor.DescriptorOperationAdapterLoader.loadOperationAdapters;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.loader.descriptor.DescriptorOperationLoader.loadOperations;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.loader.descriptor.DescriptorPaginationLoader.loadPaginations;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.loader.descriptor.DescriptorSecuritySchemeLoader.loadSecuritySchemes;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.loader.descriptor.DescriptorTestConnectionLoader.loadTestConnection;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.loader.descriptor.DescriptorTriggerLoader.loadTriggers;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.loader.descriptor.DescriptorTypeLoader.loadMediaType;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorCategory;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.QueryParamArrayFormat;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.BaseUriBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.ConnectorModelBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.uri.BaseUri;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.BaseUriDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.MavenGavDescriptor;

public class DescriptorConnectorModelLoader {

  public static ConnectorModelBuilder load(ConnectorDescriptor connectorDescriptor,
                                           ConnectorModelBuilder connectorModelBuilder) {

    if (connectorDescriptor.getIgnoreOperations() != null) {
      connectorModelBuilder.setAllOperationsIgnored(connectorDescriptor.getIgnoreOperations());
    }

    connectorModelBuilder
        .name(connectorDescriptor.getConnectorName())
        .description(connectorDescriptor.getConnectorDescription())
        .baseJavaPackage(connectorDescriptor.getBaseJavaPackage())
        .category(loadConnectorCategory(connectorDescriptor))
        .extensionXml(connectorDescriptor.getExtensionXml())
        .skipOutputTypeValidation(connectorDescriptor.getSkipOutputTypeValidation())
        .queryParamArrayFormat(loadQueryParamArrayFormat(connectorDescriptor.getQueryParamArrayFormat()))
        .operationIdentifierExpression(loadOperationIdentifierExpression(connectorDescriptor))
        .operationDisplayNameExpression(loadDisplayNameExpression(connectorDescriptor))
        .parameterIdentifierExpression(loadParameterIdentifierExpression(connectorDescriptor))
        .bodyIdentifierExpression(loadBodyIdentifierExpression(connectorDescriptor))
        .defaultInputMediaType(loadMediaType(connectorDescriptor.getDefaultInputMediaType()))
        .defaultOutputMediaType(loadMediaType(connectorDescriptor.getDefaultOutputMediaType()))
        .testConnection(loadTestConnection(connectorDescriptor.getTestConnection()));

    loadGav(connectorDescriptor, connectorModelBuilder);
    loadBaseUri(connectorDescriptor.getBaseUri(), connectorModelBuilder.getBaseUriBuilder());
    loadOperations(connectorDescriptor, connectorModelBuilder);
    loadOperationAdapters(connectorDescriptor, connectorModelBuilder);
    loadPaginations(connectorDescriptor, connectorModelBuilder);
    loadTriggers(connectorDescriptor, connectorModelBuilder);
    loadSecuritySchemes(connectorDescriptor, connectorModelBuilder);

    return connectorModelBuilder;
  }

  private static void loadGav(ConnectorDescriptor connectorDescriptor, ConnectorModelBuilder connectorModelBuilder) {
    MavenGavDescriptor connectorGav = connectorDescriptor.getConnectorGav();

    if (connectorGav == null) {
      return;
    }

    connectorModelBuilder
        .mavenGroup(connectorGav.getGroupId())
        .mavenArtifactId(connectorGav.getArtifactId())
        .mavenVersion(connectorGav.getVersion());
  }

  private static String loadOperationIdentifierExpression(ConnectorDescriptor connectorDescriptor) {
    return connectorDescriptor.getOperationIdentifier() != null ? connectorDescriptor.getOperationIdentifier().getExpression()
        : null;
  }

  private static String loadDisplayNameExpression(ConnectorDescriptor connectorDescriptor) {
    return connectorDescriptor.getOperationDisplayName() != null ? connectorDescriptor.getOperationDisplayName().getExpression()
        : null;
  }

  private static String loadParameterIdentifierExpression(ConnectorDescriptor connectorDescriptor) {
    return connectorDescriptor.getParameterIdentifier() != null ? connectorDescriptor.getParameterIdentifier().getExpression()
        : null;
  }

  private static String loadBodyIdentifierExpression(ConnectorDescriptor connectorDescriptor) {
    return connectorDescriptor.getBodyIdentifier() != null ? connectorDescriptor.getBodyIdentifier().getExpression()
        : null;
  }

  private static QueryParamArrayFormat loadQueryParamArrayFormat(String format) {
    if (isNotBlank(format)) {
      return QueryParamArrayFormat.valueOf(format.toUpperCase());
    }

    return null;
  }

  private static void loadBaseUri(BaseUriDescriptor baseUriDescriptor, BaseUriBuilder baseUriBuilder) {
    if (baseUriDescriptor != null) {
      baseUriBuilder
          .setUri(baseUriDescriptor.getValue())
          .setType(BaseUri.Type.valueOfName(baseUriDescriptor.getType()));

      baseUriDescriptor.getMapping()
          .forEach(x -> baseUriBuilder.addMultipleBaseUri(x.getName(), x.getValue(), x.isDefault()));
    }
  }

  private static ConnectorCategory loadConnectorCategory(ConnectorDescriptor connectorDescriptor) {
    return ConnectorCategory.fromString(connectorDescriptor.getConnectorCategory());
  }
}
