/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.connectormodel.loader.api;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.loader.api.ApiParameterLoader.loadSecuritySchemeParameters;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.HEADER;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.QUERY;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.OAuth2Scheme.GrantType.AUTHORIZATION_CODE;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.OAuth2Scheme.GrantType.CLIENT_CREDENTIALS;
import static com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APISecuritySchemeType.OAUTH2_AUTHORIZATION_CODE;
import static com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APISecuritySchemeType.OAUTH2_CLIENT_CREDENTIALS;
import static java.util.stream.Collectors.toList;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.ConnectorModelBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.SecuritySchemeBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.ConnectorSecurityScheme;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APISecuritySchemeModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APISecuritySchemeType;

import java.util.List;

public class ApiSecuritySchemeLoader {

  static void loadSecuritySchemes(APIModel apiModel, ConnectorModelBuilder connectorModelBuilder) {
    List<APISecuritySchemeModel> securitySchemes =
        apiModel.getOperationsModel().stream()
            .flatMap(x -> x.getSecuritySchemesModel().stream())
            .collect(toList());

    for (APISecuritySchemeModel apiSecuritySchemeModel : securitySchemes) {
      SecuritySchemeBuilder securitySchemeBuilder =
          connectorModelBuilder.getOrCreateSecuritySchemeBuilder(apiSecuritySchemeModel.getName(),
                                                                 getConnectorSecuritySchemeType(apiSecuritySchemeModel
                                                                     .getSecuritySchemeType()));

      loadSecurityScheme(apiSecuritySchemeModel, securitySchemeBuilder);
    }
  }

  private static void loadSecurityScheme(APISecuritySchemeModel apiSecuritySchemeModel,
                                         SecuritySchemeBuilder securitySchemeBuilder) {

    if (apiSecuritySchemeModel.getSecuritySchemeType().equals(OAUTH2_AUTHORIZATION_CODE)) {
      securitySchemeBuilder.grantType(AUTHORIZATION_CODE);
    } else if (apiSecuritySchemeModel.getSecuritySchemeType().equals(OAUTH2_CLIENT_CREDENTIALS)) {
      securitySchemeBuilder.grantType(CLIENT_CREDENTIALS);
    }

    securitySchemeBuilder.accessTokenUri(apiSecuritySchemeModel.getAccessTokenUri());
    securitySchemeBuilder.authorizationUri(apiSecuritySchemeModel.getAuthorizationUri());
    securitySchemeBuilder.scopes(apiSecuritySchemeModel.getScopes());

    loadSecuritySchemeParameters(apiSecuritySchemeModel.getCustomHeaders(), HEADER, securitySchemeBuilder);
    loadSecuritySchemeParameters(apiSecuritySchemeModel.getCustomQueryParams(), QUERY, securitySchemeBuilder);
  }

  private static ConnectorSecurityScheme.SecuritySchemeType getConnectorSecuritySchemeType(APISecuritySchemeType apiSecuritySchemeType) {
    switch (apiSecuritySchemeType) {
      case BASIC_AUTH:
        return ConnectorSecurityScheme.SecuritySchemeType.BASIC;
      case CUSTOM:
        return ConnectorSecurityScheme.SecuritySchemeType.CUSTOM_AUTHENTICATION;
      case DIGEST:
        return ConnectorSecurityScheme.SecuritySchemeType.DIGEST_AUTHENTICATION;
      case PASS_THROUGH:
        return ConnectorSecurityScheme.SecuritySchemeType.PASS_THROUGH;
      case OAUTH2_AUTHORIZATION_CODE:
      case OAUTH2_CLIENT_CREDENTIALS:
        return ConnectorSecurityScheme.SecuritySchemeType.OAUTH2;
      case BEARER_AUTH:
        return ConnectorSecurityScheme.SecuritySchemeType.BEARER;
      case NOT_SUPPORTED:
        return ConnectorSecurityScheme.SecuritySchemeType.UNSECURED;
      default:
        throw new IllegalArgumentException("API Security scheme '" + apiSecuritySchemeType
            + "' is not supported. This is a bug.");
    }
  }
}
