/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.OAuth2Scheme.GrantType.AUTHORIZATION_CODE;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.OAuth2Scheme.GrantType.CLIENT_CREDENTIALS;
import static java.util.Objects.requireNonNull;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.ObjectUtils.defaultIfNull;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.TypeSchemaPool;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dw.ParameterIdentifierExpressionHandler;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.BasicAuthDefaultParameters;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.BearerAuthDefaultParameters;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.DigestAuthDefaultParameters;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.Parameter;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.SecurityDefaultParameters;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.BasicAuthScheme;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.BearerAuthScheme;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.ConnectorSecurityScheme;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.CustomAuthenticationScheme;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.DigestAuthenticationScheme;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.OAuth2AuthorizationCodeScheme;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.OAuth2ClientCredentialsScheme;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.OAuth2Scheme;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.PassThroughScheme;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.TestConnectionConfig;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.UnsecuredScheme;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.util.ComparisonUtil;


import java.util.ArrayList;
import java.util.List;

public class SecuritySchemeBuilder {

  private final ConnectorSecurityScheme.SecuritySchemeType type;
  private final String name;

  private final List<ParameterBuilder> nonStandardHeaderBuilders = new ArrayList<>();
  private final List<ParameterBuilder> nonStandardQueryParameterBuilders = new ArrayList<>();

  private final SecurityDefaultParametersBuilder securityDefaultParametersBuilder = new SecurityDefaultParametersBuilder();

  private TestConnectionConfig testConnectionConfig;
  private String authorizationUri;
  private String accessTokenUri;
  private List<String> scopes;
  private String refreshTokenConditionExpression;
  private OAuth2Scheme.GrantType grantType;
  private Boolean ignored;

  public SecuritySchemeBuilder(String name, ConnectorSecurityScheme.SecuritySchemeType type) {
    requireNonNull(name);
    requireNonNull(type);

    this.type = type;
    this.name = name;
  }

  public String getName() {
    return name;
  }

  public ConnectorSecurityScheme.SecuritySchemeType getType() {
    return type;
  }

  public Boolean isIgnored() {
    return ignored;
  }

  public ParameterBuilder getOrCreateParameterBuilder(ParameterType parameterType, String name) {
    // TODO code smell, quite similar to
    // com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.OperationBuilder.getOrCreateParameterBuilder
    List<ParameterBuilder> parameterBuilders = getParameterBuilders(parameterType);

    ParameterBuilder paramBuilder = parameterBuilders.stream()
        .filter(x -> ComparisonUtil.externalNameParamsComparison(x.getExternalName(), name, parameterType))
        .findFirst().orElse(null);

    if (paramBuilder == null) {
      String parameterIdentifier = null; // No parameter identifier for a schema type of parameter
      paramBuilder = new ParameterBuilder(parameterType, name, parameterIdentifier);
      parameterBuilders.add(paramBuilder);
    }

    return paramBuilder;
  }

  private List<ParameterBuilder> getParameterBuilders(ParameterType parameterType) {
    switch (parameterType) {
      case QUERY:
        return nonStandardQueryParameterBuilders;
      case HEADER:
        return nonStandardHeaderBuilders;
      default:
        throw new IllegalArgumentException("Parameter Type not supported");
    }
  }


  public SecuritySchemeBuilder testConnectionConfig(TestConnectionConfig testConnectionConfig) {
    this.testConnectionConfig = defaultIfNull(testConnectionConfig, this.testConnectionConfig);
    return this;
  }

  public SecurityDefaultParametersBuilder getSecurityDefaultParametersBuilder() {
    return securityDefaultParametersBuilder;
  }

  public SecuritySchemeBuilder authorizationUri(String authorizationUri) {
    this.authorizationUri = defaultIfNull(authorizationUri, this.authorizationUri);
    return this;
  }

  public SecuritySchemeBuilder accessTokenUri(String accessTokenUri) {
    this.accessTokenUri = defaultIfNull(accessTokenUri, this.accessTokenUri);
    return this;
  }

  public SecuritySchemeBuilder scopes(List<String> scopes) {
    this.scopes = defaultIfNull(scopes, this.scopes);
    return this;
  }

  public SecuritySchemeBuilder refreshTokenConditionExpression(String refreshTokenConditionExpression) {
    this.refreshTokenConditionExpression = defaultIfNull(refreshTokenConditionExpression, this.refreshTokenConditionExpression);
    return this;
  }

  public SecuritySchemeBuilder grantType(OAuth2Scheme.GrantType grantType) {
    this.grantType = defaultIfNull(grantType, this.grantType);
    return this;
  }

  public SecuritySchemeBuilder ignored(Boolean ignored) {
    this.ignored = defaultIfNull(ignored, this.ignored);
    return this;
  }

  public ConnectorSecurityScheme build(TypeSchemaPool typeSchemaPool, TestConnectionConfig defaultTestConnectionConfig,
                                       ParameterIdentifierExpressionHandler parameterIdentifierExpressionHandler) {
    TestConnectionConfig testConnectionConfig = defaultIfNull(this.testConnectionConfig, defaultTestConnectionConfig);

    switch (type) {
      case BASIC:
        return buildBasicAuthScheme(testConnectionConfig);
      case CUSTOM_AUTHENTICATION:
        return buildCustomAuthenticationScheme(typeSchemaPool, parameterIdentifierExpressionHandler,
                                               testConnectionConfig);
      case DIGEST_AUTHENTICATION:
        return buildDigestAuthenticationSchemeScheme(testConnectionConfig);
      case PASS_THROUGH:
        return buildPassThroughScheme(typeSchemaPool, parameterIdentifierExpressionHandler,
                                      testConnectionConfig);
      case OAUTH2:
        return buildOauth2Scheme(typeSchemaPool, parameterIdentifierExpressionHandler,
                                 testConnectionConfig);
      case BEARER:
        return buildBearerAuthScheme(testConnectionConfig);
      case UNSECURED:
      case NOT_SUPPORTED:
        return buildUnsecuredScheme(testConnectionConfig);
      default:
        throw new IllegalArgumentException("Security scheme '" + type + "' is not supported.");
    }
  }

  private ConnectorSecurityScheme buildOauth2Scheme(TypeSchemaPool typeSchemaPool,
                                                    ParameterIdentifierExpressionHandler parameterIdentifierExpressionHandler,
                                                    TestConnectionConfig testConnectionConfig) {
    requireNonNull(grantType);

    if (grantType.equals(AUTHORIZATION_CODE)) {
      return buildOAuth2AuthorizationCodeScheme(typeSchemaPool, parameterIdentifierExpressionHandler,
                                                testConnectionConfig);
    } else if (grantType.equals(CLIENT_CREDENTIALS)) {
      return buildOAuth2ClientCredentialsScheme(typeSchemaPool, parameterIdentifierExpressionHandler,
                                                testConnectionConfig);
    }

    throw new IllegalArgumentException("Grant type not supported. This is a bug.");
  }

  private BasicAuthScheme buildBasicAuthScheme(TestConnectionConfig testConnectionConfig) {
    SecurityDefaultParameters securityDefaultParameters = securityDefaultParametersBuilder.build();
    if (securityDefaultParameters == null || securityDefaultParameters instanceof BasicAuthDefaultParameters) {
      return new BasicAuthScheme(name, (BasicAuthDefaultParameters) securityDefaultParameters, testConnectionConfig);
    }
    throw new IllegalArgumentException("Default parameters not corresponding to the security scheme type: basic. This is a bug.");
  }

  private CustomAuthenticationScheme buildCustomAuthenticationScheme(TypeSchemaPool typeSchemaPool,
                                                                     ParameterIdentifierExpressionHandler parameterIdentifierExpressionHandler,
                                                                     TestConnectionConfig testConnectionConfig) {
    return new CustomAuthenticationScheme(name,
                                          buildNonStandardQueryParameters(typeSchemaPool,
                                                                          parameterIdentifierExpressionHandler),
                                          buildNonStandardHeaders(typeSchemaPool,
                                                                  parameterIdentifierExpressionHandler),
                                          testConnectionConfig);
  }

  private DigestAuthenticationScheme buildDigestAuthenticationSchemeScheme(TestConnectionConfig testConnectionConfig) {
    SecurityDefaultParameters securityDefaultParameters = securityDefaultParametersBuilder.build();
    if (securityDefaultParameters == null || securityDefaultParameters instanceof DigestAuthDefaultParameters) {
      return new DigestAuthenticationScheme(name, (DigestAuthDefaultParameters) securityDefaultParameters, testConnectionConfig);
    }
    throw new IllegalArgumentException("Default parameters not corresponding to the security scheme type: digest. This is a bug.");
  }

  private OAuth2AuthorizationCodeScheme buildOAuth2AuthorizationCodeScheme(TypeSchemaPool typeSchemaPool,
                                                                           ParameterIdentifierExpressionHandler parameterIdentifierExpressionHandler,
                                                                           TestConnectionConfig testConnectionConfig) {
    return new OAuth2AuthorizationCodeScheme(name,
                                             authorizationUri,
                                             accessTokenUri,
                                             scopes,
                                             buildNonStandardQueryParameters(typeSchemaPool,
                                                                             parameterIdentifierExpressionHandler),
                                             buildNonStandardHeaders(typeSchemaPool,
                                                                     parameterIdentifierExpressionHandler),
                                             testConnectionConfig,
                                             refreshTokenConditionExpression);
  }

  private OAuth2ClientCredentialsScheme buildOAuth2ClientCredentialsScheme(TypeSchemaPool typeSchemaPool,
                                                                           ParameterIdentifierExpressionHandler parameterIdentifierExpressionHandler,
                                                                           TestConnectionConfig testConnectionConfig) {
    return new OAuth2ClientCredentialsScheme(name,
                                             authorizationUri,
                                             accessTokenUri,
                                             scopes,
                                             buildNonStandardQueryParameters(typeSchemaPool,
                                                                             parameterIdentifierExpressionHandler),
                                             buildNonStandardHeaders(typeSchemaPool,
                                                                     parameterIdentifierExpressionHandler),
                                             testConnectionConfig, refreshTokenConditionExpression);
  }

  private PassThroughScheme buildPassThroughScheme(TypeSchemaPool typeSchemaPool,
                                                   ParameterIdentifierExpressionHandler parameterIdentifierExpressionHandler,
                                                   TestConnectionConfig testConnectionConfig) {
    return new PassThroughScheme(name,
                                 buildNonStandardQueryParameters(typeSchemaPool,
                                                                 parameterIdentifierExpressionHandler),
                                 buildNonStandardHeaders(typeSchemaPool,
                                                         parameterIdentifierExpressionHandler),
                                 testConnectionConfig);
  }

  private UnsecuredScheme buildUnsecuredScheme(TestConnectionConfig testConnectionConfig) {
    return new UnsecuredScheme(testConnectionConfig);
  }

  private BearerAuthScheme buildBearerAuthScheme(TestConnectionConfig testConnectionConfig) {
    SecurityDefaultParameters securityDefaultParameters = securityDefaultParametersBuilder.build();
    if (securityDefaultParameters == null || securityDefaultParameters instanceof BearerAuthDefaultParameters) {
      return new BearerAuthScheme(name, (BearerAuthDefaultParameters) securityDefaultParameters, testConnectionConfig);
    }
    throw new IllegalArgumentException("Default parameters not corresponding to the security scheme type: bearer. This is a bug.");
  }

  private List<Parameter> buildNonStandardQueryParameters(TypeSchemaPool typeSchemaPool,
                                                          ParameterIdentifierExpressionHandler parameterIdentifierExpressionHandler) {
    return nonStandardQueryParameterBuilders.stream()
        .filter(x -> x.isIgnored() == null || !x.isIgnored())
        .map(x -> x.buildParameter(typeSchemaPool, parameterIdentifierExpressionHandler))
        .collect(toList());
  }

  private List<Parameter> buildNonStandardHeaders(TypeSchemaPool typeSchemaPool,
                                                  ParameterIdentifierExpressionHandler parameterIdentifierExpressionHandler) {
    return nonStandardHeaderBuilders.stream()
        .filter(x -> x.isIgnored() == null || !x.isIgnored())
        .map(x -> x.buildParameter(typeSchemaPool, parameterIdentifierExpressionHandler))
        .collect(toList());
  }
}

