/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.connectormodel.loader.descriptor;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.loader.descriptor.DescriptorSampleDataLoader.loadSampleData;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.loader.descriptor.DescriptorValueProviderLoader.loadValueProvider;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.BODY;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.HEADER;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.QUERY;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.URI;
import static java.util.Optional.ofNullable;
import static org.apache.commons.lang3.StringUtils.isEmpty;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.AuxiliarParameterBindingBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.AuxiliarParameterBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.BodyBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.ConnectorModelBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.FieldBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.OperationBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.ParameterBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.TypeDefinitionBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.generic.ParameterDataType;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.AuxiliarBodyBindingDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.AuxiliarParameterBindingDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.AuxiliarParameterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.AuxiliarParameterRequestBindingsDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.AuxiliarParameterResponseBindingsDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.DataTypeDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.FieldDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationAdapterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.BaseEndpointDescriptor;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.Optional;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import javax.ws.rs.core.MediaType;

import org.apache.commons.lang3.StringUtils;

public class DescriptorOperationAdapterLoader {

  static void loadOperationAdapters(ConnectorDescriptor connectorDescriptor, ConnectorModelBuilder connectorModelBuilder) {
    for (OperationAdapterDescriptor operationAdapterDescriptor : connectorDescriptor.getOperationAdapterDescriptors()) {
      loadOperationAdapter(operationAdapterDescriptor, connectorModelBuilder);
    }
  }

  private static void loadOperationAdapter(OperationAdapterDescriptor operationAdapterDescriptor,
                                           ConnectorModelBuilder connectorModelBuilder) {
    if (StringUtils.isNotBlank(operationAdapterDescriptor.getFqn())) {
      connectorModelBuilder.createNativeOperationAdapterBuilder(
                                                                operationAdapterDescriptor.getOperationId(),
                                                                operationAdapterDescriptor.getFqn());

    } else {
      OperationBuilder baseOperationBuilder = resolveBaseOperationBuilder(operationAdapterDescriptor, connectorModelBuilder);
      final OperationBuilder operationBuilder =
          connectorModelBuilder.createOperationAdapterBuilder(operationAdapterDescriptor.getOperationId(),
                                                              baseOperationBuilder);
      loadOperationAdapter(operationBuilder, operationAdapterDescriptor);
    }
  }

  private static OperationBuilder resolveBaseOperationBuilder(OperationAdapterDescriptor operationAdapterDescriptor,
                                                              ConnectorModelBuilder connectorModelBuilder) {

    final BaseEndpointDescriptor baseEndpoint = operationAdapterDescriptor.getBaseEndpoint();
    final OperationBuilder operationBuildersByOperationId =
        connectorModelBuilder.getOperationBuildersByOperationId(baseEndpoint)
            .orElseThrow(() -> new IllegalArgumentException(String.format("Base operation not found %s, this is a bug.",
                                                                          baseEndpoint.toString())));
    if (!operationBuildersByOperationId.isAdapter()) {
      return operationBuildersByOperationId;
    } else {
      throw new IllegalArgumentException(String.format("Base operation %s is not an endpoint operation.",
                                                       baseEndpoint.toString()));
    }
  }

  private static void loadOperationAdapter(OperationBuilder operationBuilder,
                                           OperationAdapterDescriptor operationAdapterDescriptor) {

    operationBuilder.ignored(false);
    ofNullable(operationAdapterDescriptor.getVoidOperation()).ifPresent(operationBuilder::voidOperation);
    operationBuilder.privateOperation(ofNullable(operationAdapterDescriptor.getPrivateOperation()).orElse(false));
    ofNullable(operationAdapterDescriptor.getDisplayName()).ifPresent(operationBuilder::displayName);
    ofNullable(operationAdapterDescriptor.getDescription()).ifPresent(operationBuilder::description);
    final List<AuxiliarParameterBindingBuilder> requestBindings =
        buildRequestBindings(operationAdapterDescriptor.getRequestBindings());
    operationBuilder.requestBindings(requestBindings);
    final List<AuxiliarParameterBindingBuilder> responseBindings =
        buildResponseBindings(operationAdapterDescriptor.getResponseBindings());
    operationBuilder.responseBindings(responseBindings);
    String muleOutputResolver = buildMuleOutputResolver(operationAdapterDescriptor.getResponseBindings());
    operationBuilder.muleOutputResolver(muleOutputResolver);
    operationBuilder.uriParameters(filterParametersAlreadyBound(operationBuilder.getUriParameters(), requestBindings));
    operationBuilder.queryParameters(filterParametersAlreadyBound(operationBuilder.getQueryParameters(), requestBindings));
    operationBuilder.headers(filterParametersAlreadyBound(operationBuilder.getHeaders(), requestBindings));
    final List<AuxiliarParameterDescriptor> parameters = operationAdapterDescriptor.getParameters();
    operationBuilder.parameters(buildParameters(parameters, operationBuilder.getBodyBuilder()));

    getBodyBinding(responseBindings).ifPresent(auxiliarParameterBindingBuilder -> {
      final TypeDefinitionBuilder typeDefinitionBuilder = auxiliarParameterBindingBuilder.getTypeDefinitionBuilder();
      MediaType mediaType = typeDefinitionBuilder.getMediaType();
      final Supplier<String> rawSchemaSupplier = typeDefinitionBuilder.getRawSchemaSupplier();
      if (mediaType != null && rawSchemaSupplier != null) {
        // TODO: Review
        operationBuilder
            .forceOutputTypeSchema(rawSchemaSupplier.get())
            .setDefaultOutputMediaType(mediaType);
        // operationBuilder.getOutputMetadataBuilder(mediaType).typeSchema();
      }
    });
    operationBuilder.adapter();
    if (ofNullable(operationAdapterDescriptor.getSidecarOperation()).orElse(false)) {
      operationBuilder.sidecar();
    }
    loadSampleData(operationAdapterDescriptor.getSampleDataExpressionDescriptor(), operationBuilder.getSampleDataBuilder());
  }

  private static List<ParameterBuilder> filterParametersAlreadyBound(List<ParameterBuilder> parameters,
                                                                     List<AuxiliarParameterBindingBuilder> requestBindings) {
    if (requestBindings == null) {
      return new ArrayList<>(parameters);
    }
    return parameters.stream().filter(parameter -> !isParameterAlreadyBound(parameter, requestBindings))
        .collect(Collectors.toList());
  }

  private static boolean isParameterAlreadyBound(ParameterBuilder parameter,
                                                 List<AuxiliarParameterBindingBuilder> requestBindings) {
    if (parameter == null) {
      throw new IllegalArgumentException();
    }
    if (requestBindings == null) {
      return false;
    }

    return requestBindings.stream().anyMatch(parameterBinding -> isSameBinding(parameter, parameterBinding));
  }

  private static Optional<AuxiliarParameterBindingBuilder> getBodyBinding(List<AuxiliarParameterBindingBuilder> requestBindings) {
    if (requestBindings == null) {
      return Optional.empty();
    }

    return requestBindings.stream().filter(DescriptorOperationAdapterLoader::isBodyBinding).findFirst();
  }

  private static boolean isSameBinding(ParameterBuilder parameter, AuxiliarParameterBindingBuilder parameterBinding) {
    return parameter.getParameterType().equals(parameterBinding.getParameterType())
        && parameter.getExternalName().equals(parameterBinding.getName());
  }

  private static boolean isBodyBinding(AuxiliarParameterBindingBuilder parameterBinding) {
    return parameterBinding.getParameterType().equals(BODY);
  }


  private static String buildMuleOutputResolver(AuxiliarParameterResponseBindingsDescriptor parameterBindings) {
    if (parameterBindings == null) {
      return null;
    }
    return parameterBindings.getMuleOutputResolver();
  }

  private static List<AuxiliarParameterBindingBuilder> buildResponseBindings(AuxiliarParameterResponseBindingsDescriptor parameterBindings) {
    if (parameterBindings == null) {
      return null;
    }

    final List<AuxiliarParameterBindingBuilder> parameterBindingList = new LinkedList<>();

    if (parameterBindings.getBodyBinding() != null) {
      parameterBindingList.add(buildBodyBinding(parameterBindings.getBodyBinding()));
    }

    return parameterBindingList;
  }

  private static List<AuxiliarParameterBindingBuilder> buildRequestBindings(AuxiliarParameterRequestBindingsDescriptor parameterBindings) {
    if (parameterBindings == null) {
      return null;
    }

    final List<AuxiliarParameterBindingBuilder> parameterBindingList = new LinkedList<>();

    if (parameterBindings.getUriParameters() != null) {
      for (AuxiliarParameterBindingDescriptor uriParam : parameterBindings.getUriParameters()) {
        parameterBindingList.add(buildParameterBinding(uriParam, URI));
      }
    }

    if (parameterBindings.getQueryParameters() != null) {
      for (AuxiliarParameterBindingDescriptor queryParam : parameterBindings.getQueryParameters()) {
        parameterBindingList.add(buildParameterBinding(queryParam, QUERY));
      }
    }

    if (parameterBindings.getHeaders() != null) {
      for (AuxiliarParameterBindingDescriptor header : parameterBindings.getHeaders()) {
        parameterBindingList.add(buildParameterBinding(header, HEADER));
      }
    }

    if (parameterBindings.getBodyBindings() != null) {
      for (AuxiliarParameterBindingDescriptor body : parameterBindings.getBodyBindings()) {
        parameterBindingList.add(buildParameterBinding(body, BODY));
      }
    }

    final String requestBodyExpression = parameterBindings.getRequestBodyExpression();
    if (requestBodyExpression != null) {
      parameterBindingList.add(buildBodyParameterBinding(requestBodyExpression));
    }

    return parameterBindingList;
  }

  private static AuxiliarParameterBindingBuilder buildBodyParameterBinding(String value) {
    AuxiliarParameterBindingBuilder auxiliarParameterBindingBuilder = new AuxiliarParameterBindingBuilder(BODY.getName(), BODY);
    auxiliarParameterBindingBuilder.value(value);
    return auxiliarParameterBindingBuilder;
  }

  private static AuxiliarParameterBindingBuilder buildBodyBinding(AuxiliarBodyBindingDescriptor descriptor) {
    AuxiliarParameterBindingBuilder auxiliarParameterBindingBuilder = new AuxiliarParameterBindingBuilder(BODY.getName(), BODY);
    auxiliarParameterBindingBuilder.value(descriptor.getValue());

    final String schemaType = descriptor.getMediaType();
    if (StringUtils.isNotBlank(schemaType)) {
      final TypeDefinitionBuilder typeDefinitionBuilder = auxiliarParameterBindingBuilder.getTypeDefinitionBuilder();
      typeDefinitionBuilder.object().typeSchema(descriptor::getMediaType);
      typeDefinitionBuilder.mediaType(loadMediaType(descriptor.getContentType()));
    }

    return auxiliarParameterBindingBuilder;
  }

  private static AuxiliarParameterBindingBuilder buildParameterBinding(AuxiliarParameterBindingDescriptor descriptor,
                                                                       ParameterType parameterType) {
    AuxiliarParameterBindingBuilder auxiliarParameterBindingBuilder =
        new AuxiliarParameterBindingBuilder(descriptor.getName(), parameterType);
    auxiliarParameterBindingBuilder.value(descriptor.getValue());
    auxiliarParameterBindingBuilder.ignored(descriptor.getIgnored());

    final String schemaType = descriptor.getInputType();
    if (StringUtils.isNotBlank(schemaType)) {
      final TypeDefinitionBuilder typeDefinitionBuilder = auxiliarParameterBindingBuilder.getTypeDefinitionBuilder();
      typeDefinitionBuilder.object().typeSchema(descriptor::getInputType);
      typeDefinitionBuilder.mediaType(loadMediaType(descriptor.getContentType()));
    }

    return auxiliarParameterBindingBuilder;
  }

  private static List<AuxiliarParameterBuilder> buildParameters(List<AuxiliarParameterDescriptor> parameterDescriptors,
                                                                BodyBuilder bodyBuilder) {
    final List<AuxiliarParameterBuilder> parameters = new LinkedList<>();
    if (parameterDescriptors != null) {
      for (AuxiliarParameterDescriptor parameterDescriptor : parameterDescriptors) {
        parameters.add(buildParameter(parameterDescriptor, bodyBuilder));
      }
    }

    return parameters;
  }

  private static AuxiliarParameterBuilder buildParameter(AuxiliarParameterDescriptor parameterDescriptor,
                                                         BodyBuilder bodyBuilder) {
    AuxiliarParameterBuilder auxiliarParameterBuilder = new AuxiliarParameterBuilder(parameterDescriptor.getName());
    auxiliarParameterBuilder
        .displayName(parameterDescriptor.getDisplayName())
        .description(parameterDescriptor.getDescription())
        .muleMetadataKeyId(parameterDescriptor.getMuleMetadataKeyId())
        .muleTypeResolver(parameterDescriptor.getMuleTypeResolver())
        .required(parameterDescriptor.isRequired());

    final DataTypeDescriptor dataType = parameterDescriptor.getDataType();
    if (dataType != null) {
      final ParameterDataType parameterDataType = ParameterDataType.forName(dataType.getName());
      auxiliarParameterBuilder.dataType(parameterDataType);
    } else {
      final TypeDefinitionBuilder typeDefinitionBuilder = auxiliarParameterBuilder.getTypeDefinitionBuilder();
      if (StringUtils.isNotBlank(parameterDescriptor.getInputType())) {
        typeDefinitionBuilder.object().typeSchema(parameterDescriptor::getInputType);
      }
      typeDefinitionBuilder.mediaType(loadMediaType(parameterDescriptor.getContentType()));
    }

    loadValueProvider(parameterDescriptor.getValueProvider(), auxiliarParameterBuilder.getValueProviderExpressionBuilder());
    loadBody(parameterDescriptor.getFieldDescriptors(), auxiliarParameterBuilder);

    return auxiliarParameterBuilder;
  }

  private static void loadBody(List<FieldDescriptor> fieldDescriptors, AuxiliarParameterBuilder auxiliarParameterBuilder) {
    for (FieldDescriptor fieldDescriptor : fieldDescriptors) {
      FieldBuilder fieldBuilder = new FieldBuilder(fieldDescriptor.getParamName());
      loadValueProvider(fieldDescriptor.getValueProviders(), fieldBuilder.getValueProviderBuilder());
      auxiliarParameterBuilder.getFieldBuilders().add(fieldBuilder);
    }
  }

  private static MediaType loadMediaType(String contentType) {
    return isEmpty(contentType) ? null : MediaType.valueOf(contentType);
  }

}
