/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.connectormodel.loader.descriptor;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.loader.descriptor.DescriptorSampleDataLoader.loadSampleData;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.loader.descriptor.DescriptorTypeLoader.loadMediaType;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.loader.descriptor.DescriptorValueProviderLoader.loadValueProvider;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.HEADER;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.QUERY;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.URI;
import static java.util.Optional.ofNullable;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.HTTPMethod;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.ConnectorModelBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.TriggerBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.TriggerParameterBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.TypeDefinitionBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.generic.ParameterDataType;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.DataTypeDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ParameterBindingDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.TriggerDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.TriggerParameterDescriptor;

import java.util.ArrayList;
import java.util.List;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import org.apache.commons.lang.StringUtils;

public class DescriptorTriggerLoader {

  static void loadTriggers(ConnectorDescriptor connectorDescriptor, ConnectorModelBuilder connectorBuilder) {
    if (connectorDescriptor.getTriggers() == null) {
      return;
    }

    List<TriggerDescriptor> nonNativeTriggers = connectorDescriptor.getTriggers()
        .stream().filter(triggerDescriptor -> StringUtils.isBlank(triggerDescriptor.getFqn()))
        .collect(Collectors.toList());

    nonNativeTriggers.stream()
        .filter(triggerDescriptor -> triggerDescriptor.getBase() == null)
        .forEach(triggerDescriptor -> loadTrigger(triggerDescriptor,
                                                  connectorBuilder.getOrCreateTriggerBuilder(triggerDescriptor.getName())));

    List<TriggerDescriptor> orderedTriggers = orderTriggersDependencies(nonNativeTriggers, connectorBuilder);

    orderedTriggers.stream()
        .filter(triggerDescriptor -> triggerDescriptor.getBase() != null)
        .forEach(triggerDescriptor -> loadTriggerAdapter(triggerDescriptor, connectorBuilder));

    connectorDescriptor.getTriggers()
        .stream()
        .filter(triggerDescriptor -> StringUtils.isNotBlank(triggerDescriptor.getFqn()))
        .forEach(triggerDescriptor -> loadNativeTrigger(triggerDescriptor,
                                                        connectorBuilder.getOrCreateTriggerBuilder(triggerDescriptor.getName())));

  }

  private static List<TriggerDescriptor> orderTriggersDependencies(List<TriggerDescriptor> nonNativeTriggers,
                                                                   ConnectorModelBuilder connectorBuilder) {
    List<TriggerDescriptor> orderedTriggers = new ArrayList<>();

    nonNativeTriggers.stream()
        .filter(triggerDescriptor -> triggerDescriptor.getBase() == null ||
            connectorBuilder.getTriggerBuilders().stream().anyMatch(x -> x.getName().equals(triggerDescriptor.getBase())))
        .forEach(triggerDescriptor -> orderedTriggers.add(triggerDescriptor));

    List<TriggerDescriptor> reducedList =
        nonNativeTriggers.stream().filter(triggerDescriptor -> orderedTriggers.stream()
            .noneMatch(y -> y.equals(triggerDescriptor))).collect(
                                                                  Collectors.toList());
    while (reducedList.size() > 0) {
      List<TriggerDescriptor> auxReduced = new ArrayList<>(reducedList);
      for (TriggerDescriptor triggerDescriptor : auxReduced) {
        TriggerDescriptor parent = orderedTriggers.stream().filter(x -> x.getName().equals(triggerDescriptor.getBase()))
            .findFirst().orElse(null);
        if (parent != null) {
          orderedTriggers.add(triggerDescriptor);
          reducedList.remove(triggerDescriptor);
        }
      }
    }
    return orderedTriggers;
  }

  private static void loadTrigger(TriggerDescriptor triggerDescriptor, TriggerBuilder triggerBuilder) {

    triggerBuilder
        .path(triggerDescriptor.getPath())
        .method(triggerDescriptor.getMethod() != null ? HTTPMethod.fromString(triggerDescriptor.getMethod().getName()) : null)
        .itemsExpression(triggerDescriptor.getItemsExpression())
        .outputMediaType(triggerDescriptor.getOutputMediaType())
        .identityExpression(triggerDescriptor.getIdentityExpression())
        .outputTypeSchema(triggerDescriptor.getOutputTypeSchema() != null ? triggerDescriptor.getOutputTypeSchema().getContent()
            : null)
        .muleMetadataScope(triggerDescriptor.getMuleMetadataScope())
        .displayName(triggerDescriptor.getDisplayName())
        .setDescription(triggerDescriptor.getDescription())
        .alias(triggerDescriptor.getAlias())
        .ignored(triggerDescriptor.getIgnored())
        .refined(triggerDescriptor.getRefined())
        .eventExpression(triggerDescriptor.getEventExpression())
        .startValue(triggerDescriptor.getStartValueExpression());

    if (triggerDescriptor.getWatermark() != null) {
      triggerBuilder.watermarkExpression(triggerDescriptor.getWatermark().getValue());

      if (triggerDescriptor.getWatermark().getDataType() != null) {
        triggerBuilder.watermarkType(ParameterDataType.forName(triggerDescriptor.getWatermark().getDataType().getName()));
      }

    }

    loadParameterBindings(triggerBuilder, triggerDescriptor);
    loadParameters(triggerBuilder, triggerDescriptor);
    loadSampleData(triggerDescriptor.getSampleDataExpressionDescriptor(), triggerBuilder.getSampleDataExpressionBuilder());
  }

  private static void loadNativeTrigger(TriggerDescriptor triggerDescriptor, TriggerBuilder triggerBuilder) {
    triggerBuilder.fqn(triggerDescriptor.getFqn());
  }

  private static void loadTriggerAdapter(TriggerDescriptor triggerAdapterDescriptor,
                                         ConnectorModelBuilder connectorModelBuilder) {

    TriggerBuilder baseTriggerBuilder = resolveBaseTriggerBuilder(triggerAdapterDescriptor, connectorModelBuilder);
    final TriggerBuilder triggerBuilder =
        connectorModelBuilder.createTriggerAdapterBuilder(triggerAdapterDescriptor.getName(), baseTriggerBuilder);

    if (ofNullable(triggerAdapterDescriptor.getSidecarTrigger()).orElse(false)) {
      triggerBuilder.sidecar();
    }
    ofNullable(triggerAdapterDescriptor.getDisplayName()).ifPresent(triggerBuilder::displayName);
    ofNullable(triggerAdapterDescriptor.getDescription()).ifPresent(triggerBuilder::setDescription);
    ofNullable(triggerAdapterDescriptor.getEventExpression()).ifPresent(triggerBuilder::eventExpression);
    ofNullable(triggerAdapterDescriptor.getStartValueExpression()).ifPresent(triggerBuilder::startValue);
    ofNullable(triggerAdapterDescriptor.getOutputMediaType()).ifPresent(triggerBuilder::outputMediaType);
    ofNullable(triggerAdapterDescriptor.getMuleMetadataScope()).ifPresent(triggerBuilder::muleMetadataScope);
    if (triggerAdapterDescriptor.getOutputTypeSchema() != null) {
      ofNullable(triggerAdapterDescriptor.getOutputTypeSchema().getContent()).ifPresent(triggerBuilder::outputTypeSchema);
    }

    loadParameters(triggerBuilder, triggerAdapterDescriptor);
    loadParameterBindings(triggerBuilder, triggerAdapterDescriptor);
    loadSampleData(triggerAdapterDescriptor.getSampleDataExpressionDescriptor(), triggerBuilder.getSampleDataExpressionBuilder());
  }

  private static TriggerBuilder resolveBaseTriggerBuilder(TriggerDescriptor triggerAdapterDescriptor,
                                                          ConnectorModelBuilder connectorModelBuilder) {

    final String baseTriggerName = triggerAdapterDescriptor.getBase();
    final TriggerBuilder triggerBuildersByTriggerName =
        connectorModelBuilder.getTriggerBuildersByName(baseTriggerName)
            .orElseThrow(() -> new IllegalArgumentException(String.format("Base trigger not found '%s'. This is a bug.",
                                                                          baseTriggerName)));
    return triggerBuildersByTriggerName;

  }

  private static void loadParameterBindings(TriggerBuilder triggerBuilder, TriggerDescriptor triggerDescriptor) {
    if (triggerDescriptor.getParameterBindings() == null) {
      return;
    }

    triggerBuilder.requestBodyExpression(triggerDescriptor.getParameterBindings().getWeaveExpression());
    loadParameterBindings(triggerDescriptor.getParameterBindings().getUriParameters(), URI, triggerBuilder);
    loadParameterBindings(triggerDescriptor.getParameterBindings().getQueryParameters(), QUERY, triggerBuilder);
    loadParameterBindings(triggerDescriptor.getParameterBindings().getHeaders(), HEADER, triggerBuilder);
  }

  private static void loadParameterBindings(List<ParameterBindingDescriptor> parameterBindings,
                                            ParameterType parameterType,
                                            TriggerBuilder triggerBuilder) {

    for (ParameterBindingDescriptor bindingDescriptor : parameterBindings) {
      triggerBuilder
          .getOrCreateParameterBindingBuilder(bindingDescriptor.getName(), parameterType)
          .value(bindingDescriptor.getValue());
    }
  }

  private static void loadParameters(TriggerBuilder triggerBuilder, TriggerDescriptor triggerDescriptor) {
    if (triggerDescriptor.getParameters() == null) {
      return;
    }

    for (TriggerParameterDescriptor parameterDescriptor : triggerDescriptor.getParameters()) {
      TriggerParameterBuilder triggerParameterBuilder =
          triggerBuilder.getOrCreateParameterBuilder(parameterDescriptor.getName());
      triggerParameterBuilder.displayName(parameterDescriptor.getDisplayName());
      triggerParameterBuilder.description(parameterDescriptor.getDescription());
      triggerParameterBuilder.muleMetadataKeyId(parameterDescriptor.getMuleMetadataKeyId());
      triggerParameterBuilder.muleTypeResolver(parameterDescriptor.getMuleTypeResolver());
      triggerParameterBuilder.muleContent(parameterDescriptor.getMuleContent());
      triggerParameterBuilder.muleAlias(parameterDescriptor.getMuleAlias());
      triggerParameterBuilder.required(parameterDescriptor.isRequired());

      final DataTypeDescriptor dataType = parameterDescriptor.getDataType();
      if (dataType != null) {
        final ParameterDataType parameterDataType = ParameterDataType.forName(dataType.getName());
        triggerParameterBuilder.dataType(parameterDataType);
      } else {
        final TypeDefinitionBuilder typeDefinitionBuilder = triggerParameterBuilder.getTypeDefinitionBuilder();
        if (isNotBlank(parameterDescriptor.getInputType().getContent())) {
          typeDefinitionBuilder.object().typeSchema(() -> parameterDescriptor.getInputType().getContent());

        }
        typeDefinitionBuilder.mediaType(loadMediaType(parameterDescriptor.getContentType()));
      }

      loadValueProvider(parameterDescriptor.getValueProvider(), triggerParameterBuilder.getValueProviderExpressionBuilder());
    }

  }
}
