/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder;

import com.mulesoft.connectivity.rest.commons.internal.util.MetadataUtils;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.HTTPMethod;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.TypeSchemaPool;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.resolver.sampledata.SampleDataBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.generic.ParameterDataType;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.operation.ConnectorOperation;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.trigger.NativeTrigger;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.trigger.Trigger;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.TypeDefinition;
import org.apache.commons.lang3.StringUtils;

import javax.ws.rs.core.MediaType;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import static java.util.Objects.requireNonNull;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.ObjectUtils.defaultIfNull;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

public class TriggerBuilder {

  public static final String DEFAULT_DW_START_VALUE = "#[now()]";

  private final String name;

  private String operationId;
  private String path;
  private HTTPMethod method;
  private String displayName;
  private String description;
  private String alias;
  private String itemsExpression;
  private String identityExpression;
  private String requestBodyExpression;
  private String watermarkExpression;
  private ParameterDataType watermarkType;
  private List<TriggerParameterBuilder> parameterBuilders = new ArrayList<>();
  private List<ParameterBindingBuilder> parameterBindingBuilders = new ArrayList<>();
  private String outputTypeSchema;
  private String outputMediaType;
  private String muleMetadataScope;
  private SampleDataBuilder sampleDataBuilder = new SampleDataBuilder();
  private Boolean ignored;
  private String eventExpression;
  private String startValue;
  private boolean isSidecar;
  private String fqn;
  private Boolean refined;

  TriggerBuilder(String name) {
    requireNonNull(name);

    this.name = name;
  }

  TriggerBuilder(String name, TriggerBuilder baseTriggerBuilder) {
    this.name = name;
    this.operationId = baseTriggerBuilder.operationId;
    this.path = baseTriggerBuilder.path;
    this.method = baseTriggerBuilder.method;
    this.displayName = baseTriggerBuilder.displayName;

    this.description = baseTriggerBuilder.description;
    this.alias = baseTriggerBuilder.alias;
    this.itemsExpression = baseTriggerBuilder.itemsExpression;
    this.identityExpression = baseTriggerBuilder.identityExpression;
    this.requestBodyExpression = baseTriggerBuilder.requestBodyExpression;
    this.watermarkExpression = baseTriggerBuilder.watermarkExpression;
    this.watermarkType = baseTriggerBuilder.watermarkType;
    this.parameterBuilders = baseTriggerBuilder.parameterBuilders.stream().map(TriggerParameterBuilder::new).collect(toList());
    this.parameterBindingBuilders =
        baseTriggerBuilder.parameterBindingBuilders.stream().map(ParameterBindingBuilder::new).collect(toList());
    this.outputTypeSchema = baseTriggerBuilder.outputTypeSchema;
    this.outputMediaType = baseTriggerBuilder.outputMediaType;
    this.muleMetadataScope = baseTriggerBuilder.muleMetadataScope;
    this.sampleDataBuilder = new SampleDataBuilder(baseTriggerBuilder.sampleDataBuilder);
    this.eventExpression = baseTriggerBuilder.eventExpression;
    this.startValue = baseTriggerBuilder.startValue;
  }

  public String getOperationId() {
    return operationId;
  }

  public String getName() {
    return name;
  }

  public String getPath() {
    return path;
  }

  public HTTPMethod getMethod() {
    return method;
  }

  public TriggerBuilder operationId(String operationId) {
    this.operationId = operationId;
    return this;
  }

  public TriggerBuilder path(String path) {
    this.path = defaultIfNull(path, this.path);
    return this;
  }

  public TriggerBuilder method(HTTPMethod method) {
    this.method = defaultIfNull(method, this.method);
    return this;
  }

  public TriggerBuilder displayName(String displayName) {
    this.displayName = defaultIfNull(displayName, this.displayName);
    return this;
  }

  public String getDisplayName() {
    return displayName;
  }

  public TriggerBuilder setDescription(String description) {
    this.description = defaultIfNull(description, this.description);
    return this;
  }

  public String getAlias() {
    return alias;
  }

  public TriggerBuilder alias(String alias) {
    this.alias = defaultIfNull(alias, this.alias);
    return this;
  }

  public TriggerBuilder itemsExpression(String itemsExpression) {
    this.itemsExpression = defaultIfNull(itemsExpression, this.itemsExpression);
    return this;
  }

  public String getItemsExpression() {
    return itemsExpression;
  }

  public TriggerBuilder identityExpression(String identityExpression) {
    this.identityExpression = defaultIfNull(identityExpression, this.identityExpression);
    return this;
  }

  public TriggerBuilder eventExpression(String eventExpression) {
    this.eventExpression = defaultIfNull(eventExpression, this.eventExpression);
    return this;
  }

  public TriggerBuilder startValue(String startValue) {
    if (StringUtils.isBlank(startValue)) {
      startValue = DEFAULT_DW_START_VALUE;
    }
    this.startValue = defaultIfNull(startValue, this.startValue);
    return this;
  }

  public TriggerBuilder requestBodyExpression(String requestBodyExpression) {
    this.requestBodyExpression = defaultIfNull(requestBodyExpression, this.requestBodyExpression);
    return this;
  }

  public TriggerBuilder watermarkExpression(String watermarkExpression) {
    this.watermarkExpression = defaultIfNull(watermarkExpression, this.watermarkExpression);
    return this;
  }

  public TriggerBuilder watermarkType(ParameterDataType watermarkType) {
    this.watermarkType = defaultIfNull(watermarkType, this.watermarkType);
    return this;
  }

  public TriggerBuilder outputTypeSchema(String outputTypeSchema) {
    this.outputTypeSchema = defaultIfNull(outputTypeSchema, this.outputTypeSchema);
    return this;
  }

  public String getOutputTypeSchema() {
    return outputTypeSchema;
  }

  public TriggerBuilder outputMediaType(String outputMediaType) {
    this.outputMediaType = defaultIfNull(outputMediaType, this.outputMediaType);
    return this;
  }

  public TriggerBuilder muleMetadataScope(String muleMetadataScope) {
    this.muleMetadataScope = muleMetadataScope;
    return this;
  }

  public String getMuleMetadataScope() {
    return muleMetadataScope;
  }

  public TriggerBuilder ignored(Boolean ignored) {
    this.ignored = defaultIfNull(ignored, this.ignored);
    return this;
  }

  public TriggerBuilder refined(Boolean refined) {
    this.refined = defaultIfNull(refined, this.refined);
    return this;
  }

  public TriggerBuilder sidecar() {
    this.isSidecar = true;
    return this;
  }

  public TriggerBuilder fqn(String fqn) {
    this.fqn = defaultIfNull(fqn, this.fqn);
    return this;
  }

  public ParameterBindingBuilder getOrCreateParameterBindingBuilder(String name, ParameterType parameterType) {
    ParameterBindingBuilder parameterBindingBuilder = parameterBindingBuilders.stream()
        .filter(x -> x.getName().equals(name) && x.getParameterType().equals(parameterType))
        .findFirst().orElse(null);

    if (parameterBindingBuilder == null) {
      parameterBindingBuilder = new ParameterBindingBuilder(name, parameterType);
      parameterBindingBuilders.add(parameterBindingBuilder);
    }

    return parameterBindingBuilder;
  }

  public TriggerParameterBuilder getOrCreateParameterBuilder(String name) {
    TriggerParameterBuilder parameterBuilder = parameterBuilders.stream()
        .filter(x -> x.getName().equalsIgnoreCase(name))
        .findFirst().orElse(null);

    if (parameterBuilder == null) {
      parameterBuilder = new TriggerParameterBuilder(name);
      parameterBuilders.add(parameterBuilder);
    }

    return parameterBuilder;
  }

  public SampleDataBuilder getSampleDataExpressionBuilder() {
    return sampleDataBuilder;
  }

  Trigger buildNative() {
    return new NativeTrigger(name, fqn);
  }

  Trigger build(List<ConnectorOperation> operations, TypeSchemaPool typeSchemaPool) {

    ConnectorOperation operation = operations.stream()
        .filter(op -> op.isAdapter() && op.getOperationIdentifier().equals(operationId)).findFirst()
        .orElse(operations.stream()
            .filter(op -> op.getHttpMethod().equals(method) && op.getPath().equals(path)).findFirst()
            .orElseThrow(() -> new IllegalArgumentException("Invalid path/method for this trigger. This is a bug.")));

    Trigger trigger = new Trigger(name,
                                  displayName,
                                  description,
                                  alias,
                                  itemsExpression,
                                  identityExpression,
                                  requestBodyExpression,
                                  watermarkExpression,
                                  buildWatermarkType(watermarkType),
                                  parameterBuilders.stream()
                                      .map(triggerParameterBuilder -> triggerParameterBuilder.build(typeSchemaPool))
                                      .collect(toList()),
                                  parameterBindingBuilders.stream().map(ParameterBindingBuilder::build)
                                      .collect(toList()),
                                  getTriggerOutputType(operation, typeSchemaPool),
                                  outputMediaType,
                                  muleMetadataScope,
                                  eventExpression,
                                  startValue,
                                  isSidecar,
                                  null,
                                  defaultIfNull(refined, false),
                                  operation);

    trigger.setSampleData(sampleDataBuilder.build(trigger));

    return trigger;
  }

  private ParameterDataType buildWatermarkType(ParameterDataType watermarkType) {
    if (watermarkType != null) {
      return watermarkType;
    }

    return ParameterDataType.STRING;
  }

  private TypeDefinition getTriggerOutputType(ConnectorOperation operation, TypeSchemaPool typeSchemaPool) {
    if (isNotBlank(muleMetadataScope)) {
      return null;
    }

    TypeDefinitionBuilder typeBuilder = new TypeDefinitionBuilder().object();

    if (isNotBlank(outputMediaType)) {
      typeBuilder.mediaType(MediaType.valueOf(outputMediaType));
    } else {
      if (operation.getOutputMetadata() != null) {
        typeBuilder.mediaType(operation.getOutputMetadata().getMediaType());
      }
    }

    if (operation.getOutputMetadata() != null && operation.getOutputMetadata().getTypeSchema() != null) {
      typeBuilder.typeSchema(() -> operation.getOutputMetadata().getTypeSchema().getRawSchema());
    }

    return typeBuilder.build(typeSchemaPool, outputTypeSchema, itemsExpression, MetadataUtils.LocationType.ITEM, false);
  }


  public boolean isIgnored() {
    return defaultIfNull(ignored, false);
  }

  public boolean isSidecar() {
    return isSidecar;
  }

  public String getFqn() {
    return this.fqn;
  }
}
