/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dw;

import static com.mulesoft.connectivity.rest.commons.internal.metadatamodel.handler.SchemaHandler.SUB_SCHEMA_LOCATION;

import org.mule.metadata.api.annotation.MetadataFormatPropertiesAnnotation;
import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.builder.ObjectTypeBuilder;
import org.mule.metadata.api.model.ArrayType;
import org.mule.metadata.api.model.IntersectionType;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.TupleType;
import org.mule.metadata.api.model.UnionType;
import org.mule.metadata.api.visitor.MetadataTypeVisitor;
import org.mule.metadata.message.api.el.TypeBindings;
import org.mule.runtime.api.metadata.ExpressionLanguageMetadataService;
import org.mule.weave.v2.el.metadata.EmptyCallBack;
import org.mule.weave.v2.el.metadata.WeaveMetadataServiceProvider;
import org.mule.weave.v2.model.ServiceManager;
import org.mule.weave.v2.runtime.DataWeaveResult;
import org.mule.weave.v2.runtime.DataWeaveScript;
import org.mule.weave.v2.runtime.ScriptingBindings;

import com.mulesoft.connectivity.rest.commons.api.dw.DWBindings;
import com.mulesoft.connectivity.rest.commons.internal.metadatamodel.RestJsonTypeLoader;
import com.mulesoft.connectivity.rest.commons.internal.metadatamodel.RestJsonTypeLoaderConfiguration;
import com.mulesoft.connectivity.rest.commons.internal.util.MetadataUtils;

public class SchemaInferenceService {

  private static final String PAYLOAD = "payload";
  private static final String LOCATION = "location";

  private static final String[] IMPLICIT_INPUTS = {PAYLOAD, LOCATION};

  private static final DataWeaveScript SCRIPT =
      ExpressionHandlerUtils.compileDataWeaveScript(SchemaInferenceService.class, "schemaInference.dwl",
                                                    IMPLICIT_INPUTS);

  private static final ExpressionLanguageMetadataService metadataService =
      (ExpressionLanguageMetadataService) new WeaveMetadataServiceProvider().getServiceDefinition().getService();

  public SchemaInferenceService() {}

  public String changeSchemaRef(String schema, String location) {
    ScriptingBindings scriptingBindings = new ScriptingBindings();
    scriptingBindings.addBinding(PAYLOAD, schema, "application/json");
    scriptingBindings.addBinding(LOCATION, location, "application/java");

    DataWeaveResult dataWeaveResult = SCRIPT.write(scriptingBindings, ServiceManager.apply());
    return dataWeaveResult.getContentAsString();
  }

  public String inferSchema(String schema, String expression, MetadataUtils.LocationType locationType) {
    RestJsonTypeLoader restJsonTypeLoader =
        new RestJsonTypeLoader(schema, RestJsonTypeLoaderConfiguration.builder().traceSubSchemaLocation().build());
    MetadataType metadataType = restJsonTypeLoader.load("id").orElse(null);
    ObjectTypeBuilder response = BaseTypeBuilder.create(MetadataFormat.JSON).objectType();
    response.addField().key(DWBindings.BODY.getBinding()).value(metadataType);
    TypeBindings typeBindings = TypeBindings.builder()
        .addBinding(PAYLOAD, metadataType)
        .addBinding(DWBindings.RESPONSE.getBinding(), response.build())
        .build();
    MetadataType outputType =
        metadataService.getOutputType(typeBindings, ExpressionHandlerUtils.curateScript(expression), new EmptyCallBack());
    return findLocation(outputType, locationType);
  }

  private String findLocation(MetadataType metadataType, MetadataUtils.LocationType locationType) {
    ObjectHolder<String> objectHolder = new ObjectHolder<>(null);
    metadataType.accept(new MetadataTypeVisitor() {


      @Override
      public void visitArrayType(ArrayType arrayType) {
        super.visitArrayType(arrayType);
        setLocation(MetadataUtils.LocationType.ITEM.equals(locationType) ? arrayType.getType() : arrayType);
      }

      @Override
      public void visitUnion(UnionType unionType) {
        super.visitUnion(unionType);
        unionType.getTypes().forEach(metadataType1 -> metadataType1.accept(this));
      }

      @Override
      public void visitIntersection(IntersectionType intersectionType) {
        super.visitIntersection(intersectionType);
        intersectionType.getTypes().forEach(metadataType1 -> metadataType1.accept(this));
      }

      @Override
      public void visitTuple(TupleType tupleType) {
        super.visitTuple(tupleType);
        tupleType.getTypes().forEach(metadataType1 -> metadataType1.accept(this));
      }

      private void setLocation(MetadataType metadataType) {
        objectHolder.set(getSubSchemaLocation(metadataType));
      }

    });
    return objectHolder.get();
  }

  private String getSubSchemaLocation(MetadataType metadataType) {
    return metadataType.getAnnotation(MetadataFormatPropertiesAnnotation.class)
        .map(MetadataFormatPropertiesAnnotation::getValue)
        .map(f -> f.get(SUB_SCHEMA_LOCATION))
        .orElse(null);
  }

}
