/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.templating.sdk;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.HEADER;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.QUERY;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.URI;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.util.JavaUtils.getJavaUpperCamelNameFromXml;
import static javax.lang.model.element.Modifier.PRIVATE;
import static javax.lang.model.element.Modifier.PROTECTED;
import static javax.lang.model.element.Modifier.PUBLIC;

import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.http.api.HttpConstants;

import com.mulesoft.connectivity.rest.commons.api.datasense.valueprovider.RestValueProvider;
import com.mulesoft.connectivity.rest.commons.api.source.RequestParameterBinding;
import com.mulesoft.connectivity.rest.commons.internal.util.RestRequestBuilder;
import com.mulesoft.connectivity.rest.commons.internal.util.RestSdkUtils;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ConnectorModelParameter;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.trigger.ParameterBinding;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.valueresolver.ValueResolver;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.valueresolver.ValueResolverParameter;
import com.mulesoft.connectivity.rest.sdk.templating.JavaTemplateEntity;
import com.mulesoft.connectivity.rest.sdk.templating.api.RestSdkRunConfiguration;
import com.mulesoft.connectivity.rest.sdk.templating.exception.TemplatingException;

import java.nio.file.Path;
import java.util.HashMap;
import java.util.Map;

import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.FieldSpec;
import com.squareup.javapoet.JavaFile;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.ParameterizedTypeName;
import com.squareup.javapoet.TypeSpec;
import com.squareup.javapoet.WildcardTypeName;

public class SdkValueProvider extends JavaTemplateEntity {

  private static final String GET_TYPED_VALUE_OR_NULL_METHOD = "getTypedValueOrNull";
  private static final String GET_PATH_TEMPLATE_METHOD = "getPathTemplate";
  private static final String GET_REQUEST_BUILDER_METHOD = "getRequestBuilder";
  private static final String GET_PARAMETER_BINDING_METHOD = "getParameterBinding";
  private static final String PARAMETER_BINDING_LOCAL_VARIABLE = "binding";
  private static final String ADD_URI_BINDING_METHOD = "addUriParamBinding";
  private static final String ADD_QUERY_BINDING_METHOD = "addQueryParamBinding";
  private static final String ADD_HEADER_BINDING_METHOD = "addHeaderBinding";
  private static final String GET_PARAMETER_VALUES_METHOD = "getParameterValues";
  private static final String PARAMETER_VALUES_LOCAL_VARIABLE = "parameterValues";


  private static final String ITEMS_EXPRESSION_FIELD = "ITEMS_EXPRESSION";
  private static final String ITEM_VALUE_EXPRESSION_FIELD = "ITEM_VALUE_EXPRESSION";
  private static final String ITEM_NAME_EXPRESSION_FIELD = "ITEM_NAME_EXPRESSION";
  private static final String PATH_TEMPLATE_FIELD = "PATH";


  private final SdkOperation sdkOperation;
  private final ConnectorModelParameter parameter;

  private final ValueResolver valueResolverModel;
  private Map<String, SdkParameter> operationParameters = null;

  public SdkValueProvider(Path outputDir, ConnectorModel connectorModel, ConnectorModelParameter parameter,
                          SdkOperation operation,
                          RestSdkRunConfiguration runConfiguration) {
    super(outputDir, connectorModel, runConfiguration);

    this.parameter = parameter;
    this.sdkOperation = operation;

    this.valueResolverModel = connectorModel.getValueResolvers().stream()
        .filter(x -> x.getName().equals(parameter.getValueResolverReference().getId()))
        .findFirst().orElse(null);
  }

  @Override
  public void applyTemplates() throws TemplatingException {
    generateValueProviderClass();
  }

  public String getPackage() {
    return connectorModel.getBasePackage() + ".internal.metadata.values";
  }

  public String getJavaClassName() {
    return sdkOperation.getJavaClassName() + getJavaUpperCamelNameFromXml(parameter.getInternalName()) + "ValueProvider";
  }

  private void generateValueProviderClass() throws TemplatingException {
    TypeSpec.Builder valueProviderClassBuilder =
        TypeSpec
            .classBuilder(getJavaClassName())
            .addModifiers(PUBLIC)
            .superclass(RestValueProvider.class)
            .addMethod(generateConstructor())
            .addMethod(generateGetPathTemplateMethod())
            .addMethod(generateGetRequestBuilderMethod())
            .addMethod(generateGetParameterBindingMethod())
            .addMethod(generateGetParameterValuesMethod());

    addClassConstants(valueProviderClassBuilder);
    addParameters(valueProviderClassBuilder);

    JavaFile.Builder javaFileBuilder = getJavaFileBuilderForClass(valueProviderClassBuilder.build(), getPackage());
    javaFileBuilder.addStaticImport(RestSdkUtils.class, GET_TYPED_VALUE_OR_NULL_METHOD);
    writeJavaFile(javaFileBuilder.build());
  }

  private MethodSpec generateConstructor() {
    MethodSpec.Builder constructorBuilder =
        MethodSpec.constructorBuilder()
            .addModifiers(PUBLIC);

    constructorBuilder.addStatement("super($L, $L, $L)",
                                    ITEMS_EXPRESSION_FIELD,
                                    ITEM_VALUE_EXPRESSION_FIELD,
                                    ITEM_NAME_EXPRESSION_FIELD);

    return constructorBuilder.build();
  }

  private MethodSpec generateGetPathTemplateMethod() {
    MethodSpec.Builder methodBuilder =
        MethodSpec.methodBuilder(GET_PATH_TEMPLATE_METHOD)
            .addModifiers(PROTECTED)
            .returns(String.class)
            .addAnnotation(Override.class);

    methodBuilder.addStatement("return $L", PATH_TEMPLATE_FIELD);

    return methodBuilder.build();
  }

  private MethodSpec generateGetRequestBuilderMethod() {
    MethodSpec.Builder methodBuilder =
        MethodSpec.methodBuilder(GET_REQUEST_BUILDER_METHOD)
            .addModifiers(PROTECTED)
            .returns(RestRequestBuilder.class)
            .addAnnotation(Override.class)
            .addParameter(String.class, "path");

    methodBuilder.addStatement("return new $T(connection.getBaseUri(), path, $T.$L)",
                               RestRequestBuilder.class,
                               HttpConstants.Method.class,
                               valueResolverModel.getMethod().name().toUpperCase());

    return methodBuilder.build();
  }

  private MethodSpec generateGetParameterBindingMethod() {
    MethodSpec.Builder methodBuilder =
        MethodSpec.methodBuilder(GET_PARAMETER_BINDING_METHOD)
            .addModifiers(PROTECTED)
            .returns(RequestParameterBinding.class)
            .addAnnotation(Override.class);

    if (valueResolverModel.getParameterBindings() != null && valueResolverModel.getParameterBindings().size() > 0) {
      methodBuilder.addStatement("$1T $2L = new $1T()",
                                 RequestParameterBinding.class,
                                 PARAMETER_BINDING_LOCAL_VARIABLE);

      for (ParameterBinding binding : valueResolverModel.getParameterBindings()) {
        methodBuilder.addStatement("$L.$L($S, $S)",
                                   PARAMETER_BINDING_LOCAL_VARIABLE,
                                   getParameterBindingAddMethodName(binding),
                                   binding.getName(),
                                   binding.getExpression());
      }

      methodBuilder.addStatement("return $1L", PARAMETER_BINDING_LOCAL_VARIABLE);
    } else {
      methodBuilder.addStatement("return new $1T()", RequestParameterBinding.class);
    }

    return methodBuilder.build();
  }

  private String getParameterBindingAddMethodName(ParameterBinding binding) {
    switch (binding.getParameterType()) {
      case QUERY:
        return ADD_QUERY_BINDING_METHOD;
      case HEADER:
        return ADD_HEADER_BINDING_METHOD;
      case URI:
        return ADD_URI_BINDING_METHOD;
    }

    throw new IllegalArgumentException("Parameter type not supported: " + binding.getParameterType());
  }

  private MethodSpec generateGetParameterValuesMethod() {
    MethodSpec.Builder methodBuilder =
        MethodSpec.methodBuilder(GET_PARAMETER_VALUES_METHOD)
            .addModifiers(PROTECTED)
            .returns(getParameterValuesMultiMapType())
            .addAnnotation(Override.class);

    if (!valueResolverModel.getParameters().isEmpty()) {
      methodBuilder.addStatement("final $T $L = new $T()",
                                 getParameterValuesMultiMapType(),
                                 PARAMETER_VALUES_LOCAL_VARIABLE,
                                 MultiMap.class);

      Map<String, SdkParameter> operationSdkParameter = getOperationValueResolverParameters();

      for (String parameterName : operationSdkParameter.keySet()) {
        SdkParameter sdkParameter = operationSdkParameter.get(parameterName);

        methodBuilder.addStatement("$L.put($S, $L($L))",
                                   PARAMETER_VALUES_LOCAL_VARIABLE,
                                   parameterName,
                                   GET_TYPED_VALUE_OR_NULL_METHOD,
                                   sdkParameter.getJavaName());
      }

      methodBuilder.addStatement("return $L", PARAMETER_VALUES_LOCAL_VARIABLE);
    } else {
      methodBuilder.addStatement("return new $T()", getParameterValuesMultiMapType());
    }

    return methodBuilder.build();
  }

  private Map<String, SdkParameter> getOperationValueResolverParameters() {
    if (operationParameters != null) {
      return operationParameters;
    }

    final Map<String, SdkParameter> operationParameters = new HashMap<>();

    for (ValueResolverParameter valueResolverParameter : valueResolverModel.getParameters()) {

      String parameterArgument = parameter.getValueResolverReference()
          .getArguments().get(valueResolverParameter.getName());

      String[] argumentValue = parameterArgument.split("\\.");

      String parameterTypeString = argumentValue[0];
      String parameterName = argumentValue[1];

      ParameterType parameterType;
      if (parameterTypeString.startsWith("uriParameter")) {
        parameterType = URI;
      } else if (parameterTypeString.startsWith("queryParameter")) {
        parameterType = QUERY;
      } else {
        parameterType = HEADER;
      }

      operationParameters.put(valueResolverParameter.getName(),
                              sdkOperation.getSdkParameter(parameterType, parameterName));
    }

    this.operationParameters = operationParameters;
    return this.operationParameters;
  }

  private ParameterizedTypeName getParameterValuesMultiMapType() {
    ParameterizedTypeName wildcardTypedValueType =
        ParameterizedTypeName.get(
                                  ClassName.get(TypedValue.class),
                                  WildcardTypeName.subtypeOf(Object.class));

    return ParameterizedTypeName.get(ClassName.get(MultiMap.class), ClassName.get(String.class), wildcardTypedValueType);
  }

  private void addClassConstants(TypeSpec.Builder valueProviderClassBuilder) {
    valueProviderClassBuilder
        .addField(getConstantStringField(ITEMS_EXPRESSION_FIELD, valueResolverModel.getItemsExpression()))
        .addField(getConstantStringField(ITEM_VALUE_EXPRESSION_FIELD, valueResolverModel.getValueExpression()))
        .addField(getConstantStringField(ITEM_NAME_EXPRESSION_FIELD, valueResolverModel.getDisplayNameExpression()))
        .addField(getConstantStringField(PATH_TEMPLATE_FIELD, valueResolverModel.getPath()));
  }

  private void addParameters(TypeSpec.Builder valueResolverClassBuilder) {
    Map<String, SdkParameter> operationSdkParameters = getOperationValueResolverParameters();

    for (String parameterName : operationSdkParameters.keySet()) {
      valueResolverClassBuilder.addField(getParameterFieldSpec(operationSdkParameters.get(parameterName)));
    }
  }

  private FieldSpec getParameterFieldSpec(SdkParameter sdkParameter) {
    FieldSpec.Builder fieldBuilder =
        FieldSpec.builder(sdkParameter.getTypeName(),
                          sdkParameter.getJavaName(),
                          PRIVATE)
            .addAnnotation(org.mule.runtime.extension.api.annotation.param.Parameter.class);

    return fieldBuilder.build();
  }
}
