/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.templating.sdk;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.HEADER;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.QUERY;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.URI;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.util.JavaUtils.getJavaUpperCamelNameFromXml;
import static javax.lang.model.element.Modifier.PRIVATE;
import static javax.lang.model.element.Modifier.PROTECTED;
import static javax.lang.model.element.Modifier.PUBLIC;

import com.mulesoft.connectivity.rest.commons.api.datasense.valueprovider.RestValueProvider;
import com.mulesoft.connectivity.rest.commons.internal.model.builder.common.OperationEvaluationContextBuilder;
import com.mulesoft.connectivity.rest.commons.internal.model.builder.valueprovider.ValueProviderResolverExpressionBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.Parameter;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dataexpression.httprequest.HttpRequestBinding;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.generic.Argument;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.resolver.ResolverExpression;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.resolver.ResolverReference;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.valueprovider.ValueProviderDefinition;
import com.mulesoft.connectivity.rest.sdk.templating.JavaTemplateEntity;
import com.mulesoft.connectivity.rest.sdk.templating.api.RestSdkRunConfiguration;
import com.mulesoft.connectivity.rest.sdk.templating.exception.TemplatingException;

import com.squareup.javapoet.CodeBlock;
import com.squareup.javapoet.FieldSpec;
import com.squareup.javapoet.JavaFile;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.TypeSpec;

import java.nio.file.Path;
import java.util.List;

public abstract class AbstractSdkValueProvider extends JavaTemplateEntity {

  protected static final String ITEMS_EXPRESSION_FIELD = "ITEMS_EXPRESSION";
  protected static final String ITEM_VALUE_EXPRESSION_FIELD = "ITEM_VALUE_EXPRESSION";
  protected static final String ITEM_NAME_EXPRESSION_FIELD = "ITEM_NAME_EXPRESSION";
  protected static final String PATH_TEMPLATE_FIELD = "PATH";

  protected final SdkOperation sdkOperation;
  protected final Parameter parameter;
  protected final ValueProviderDefinition definition;

  public AbstractSdkValueProvider(Path outputDir,
                                  ConnectorModel connectorModel,
                                  Parameter parameter,
                                  SdkOperation operation,
                                  RestSdkRunConfiguration runConfiguration) {

    super(outputDir, connectorModel, runConfiguration);

    this.parameter = parameter;
    this.sdkOperation = operation;

    ResolverExpression<ValueProviderDefinition> valueProviderExpression = parameter.getValueProvider();

    this.definition = getValueProviderDefinition(valueProviderExpression);
  }

  private ValueProviderDefinition getValueProviderDefinition(ResolverExpression<ValueProviderDefinition> valueProviderExpression) {

    if (valueProviderExpression instanceof ValueProviderDefinition) {
      return (ValueProviderDefinition) valueProviderExpression;
    } else if (valueProviderExpression instanceof ResolverReference) {
      ResolverReference<ValueProviderDefinition> valueProviderReference =
          (ResolverReference<ValueProviderDefinition>) valueProviderExpression;

      return valueProviderReference.getDeclaration().getResolverDefinition();
    }

    throw new IllegalArgumentException("Invalid valueProviderExpression. This is a bug.");
  }

  public String getPackage() {
    return connectorModel.getBasePackage() + ".internal.metadata.values";
  }

  public String getJavaClassName() {
    return sdkOperation.getJavaClassName() + getJavaUpperCamelNameFromXml(parameter.getInternalName()) + "ValueProvider";
  }

  @Override
  public void applyTemplates() throws TemplatingException {
    if (this.definition != null) {
      generateValueProviderClass();
    }
  }

  private void generateValueProviderClass() throws TemplatingException {
    TypeSpec.Builder valueProviderClassBuilder = getValueProviderClassBuilder();

    buildValueProviderClass(valueProviderClassBuilder);

    JavaFile.Builder javaFileBuilder = getJavaFileBuilderForClass(valueProviderClassBuilder.build(), getPackage());
    writeJavaFile(javaFileBuilder.build());
  }

  private TypeSpec.Builder getValueProviderClassBuilder() {
    TypeSpec.Builder valueProviderClassBuilder =
        TypeSpec
            .classBuilder(getJavaClassName())
            .addModifiers(PUBLIC)
            .superclass(getSuperClass());

    generateBuildMethod(valueProviderClassBuilder);

    if (needsConfigureEvaluationContextMethod()) {
      valueProviderClassBuilder.addMethod(generateConfigureEvaluationContextMethod());
    }

    if (needsClassConstants()) {
      addClassConstants(valueProviderClassBuilder);
    }

    addParameterIfNeeded(valueProviderClassBuilder);

    return valueProviderClassBuilder;
  }

  protected boolean needsClassConstants() {
    return true;
  }

  protected boolean needsConfigureEvaluationContextMethod() {
    return true;
  }

  /**
   * Returns the super class the generated value provider will extend.
   * This is useful when multiple value providers extend a commons parent one where the logic is implemented. i.e. Reference.
   */
  protected TypeName getSuperClass() {
    return TypeName.get(RestValueProvider.class);
  }

  /**
   * Allows extending the default value provider class building
   */
  protected void buildValueProviderClass(TypeSpec.Builder valueProviderClassBuilder) {
    //Do nothing by default
  }

  /**
   * Generates the 'build' method of the value provider that overrides the abstract one from {@link RestValueProvider}.
   * This method creates the actual provider expression, being it an inline definition
   * or a reference to a value provider expression defined in the global context depending on the case.
   *
   * If the returned CodeBlock is null, no build method will be generated.
   */
  protected abstract CodeBlock generateBuildMethodBody();

  /**
   * Indicates if a {@link org.mule.runtime.extension.api.annotation.param.Parameter} from the parent operation of the
   * value provider is necessary for this value provider to resolve the available values.
   * If this method says is necessary, a @Parameter will be generated in the value provider class.
   * When that @Parameter is generated in the value provider, it is mandatory for it to have an assigned value in the
   * DSL for the resolver to be executed.
   *
   * @param parameterType The kind of parameter that is being checked
   * @param sdkParameter The parameter that is being checked
   */
  protected abstract boolean isBoundParameter(ParameterType parameterType, SdkParameter sdkParameter);

  protected String getBindingMethod(ParameterType parameterType) {
    switch (parameterType) {
      case URI:
        return "uriParameter";
      case QUERY:
        return "queryParameter";
      case HEADER:
        return "header";
      default:
        throw new IllegalArgumentException();
    }
  }

  private void generateBuildMethod(TypeSpec.Builder valueProviderClassBuilder) {
    CodeBlock methodBody = generateBuildMethodBody();

    if (methodBody != null) {
      MethodSpec.Builder buildMethodBuilder =
          MethodSpec.methodBuilder("build")
              .addModifiers(PROTECTED)
              .addParameter(TypeName.get(ValueProviderResolverExpressionBuilder.class), "builder")
              .addAnnotation(Override.class);

      buildMethodBuilder.addStatement(methodBody);

      valueProviderClassBuilder.addMethod(buildMethodBuilder.build());
    }
  }

  protected CodeBlock.Builder getValueProviderExpressionBuilder() {
    CodeBlock.Builder codeBlockBuilder = CodeBlock.builder();
    codeBlockBuilder.add(
                         "builder.definition(definitionBuilder -> definitionBuilder.httpRequest(httpRequestBuilder -> httpRequestBuilder.path($1L).method($2S)",
                         PATH_TEMPLATE_FIELD, definition.getRequest().getMethod());

    HttpRequestBinding bindings = definition.getRequest().getHttpRequestBinding();
    if (bindings != null) {
      codeBlockBuilder.add(".bindings(bindingBuilder -> bindingBuilder");
      bindings.getHeader().forEach(binding -> codeBlockBuilder.add(getBindingCode(binding, HEADER)));
      bindings.getQueryParameter().forEach(binding -> codeBlockBuilder.add(getBindingCode(binding, QUERY)));
      bindings.getUriParameter().forEach(binding -> codeBlockBuilder.add(getBindingCode(binding, URI)));
      codeBlockBuilder.add(")");
    }
    codeBlockBuilder.add(")");

    codeBlockBuilder.add(".itemExtractionExpression($1L).itemValueExpression($2L).itemNameExpression($3L))",
                         ITEMS_EXPRESSION_FIELD,
                         ITEM_VALUE_EXPRESSION_FIELD,
                         ITEM_NAME_EXPRESSION_FIELD);

    return codeBlockBuilder;
  }

  private CodeBlock getBindingCode(Argument binding, ParameterType parameterType) {
    CodeBlock.Builder bindingCodeBuilder = CodeBlock.builder();

    bindingCodeBuilder
        .add(".$1L($2S, argumentBuilder -> argumentBuilder.value(expressionBuilder -> expressionBuilder.expression($3S)))",
             getBindingMethod(parameterType),
             binding.getName(),
             binding.getValue().getValue());

    return bindingCodeBuilder.build();
  }

  private void addClassConstants(TypeSpec.Builder valueProviderClassBuilder) {
    valueProviderClassBuilder
        .addField(getConstantStringField(ITEMS_EXPRESSION_FIELD, definition.getItemExtractionExpression()))
        .addField(getConstantStringField(ITEM_VALUE_EXPRESSION_FIELD, definition.getItemValueExpression()))
        .addField(getConstantStringField(ITEM_NAME_EXPRESSION_FIELD, definition.getItemDisplayNameExpression()))
        .addField(getConstantStringField(PATH_TEMPLATE_FIELD, definition.getRequest().getPath()));
  }

  private void addParameterIfNeeded(TypeSpec.Builder valueProviderClassBuilder) {
    addParameterIfNeeded(valueProviderClassBuilder, URI, sdkOperation.getAllPathParameters());
    addParameterIfNeeded(valueProviderClassBuilder, QUERY, sdkOperation.getAllQueryParameters());
    addParameterIfNeeded(valueProviderClassBuilder, HEADER, sdkOperation.getAllHeaders());
  }

  private void addParameterIfNeeded(TypeSpec.Builder valueProviderClassBuilder, ParameterType parameterType,
                                    List<SdkParameter> sdkParameters) {
    for (SdkParameter sdkParameter : sdkParameters) {
      if (isBoundParameter(parameterType, sdkParameter)) {
        valueProviderClassBuilder.addField(getParameterFieldSpec(sdkParameter));
      }
    }
  }

  private FieldSpec getParameterFieldSpec(SdkParameter sdkParameter) {
    FieldSpec.Builder fieldBuilder =
        FieldSpec.builder(sdkParameter.getTypeName(),
                          sdkParameter.getJavaName(),
                          PRIVATE)
            .addAnnotation(org.mule.runtime.extension.api.annotation.param.Parameter.class);

    return fieldBuilder.build();
  }

  private MethodSpec generateConfigureEvaluationContextMethod() {
    MethodSpec.Builder methodBuilder =
        MethodSpec.methodBuilder("configureEvaluationContext")
            .addModifiers(PROTECTED)
            .addParameter(TypeName.get(OperationEvaluationContextBuilder.class), "builder")
            .addAnnotation(Override.class);

    addOperationParameterBinding(methodBuilder, sdkOperation.getAllPathParameters(), URI);
    addOperationParameterBinding(methodBuilder, sdkOperation.getAllQueryParameters(), QUERY);
    addOperationParameterBinding(methodBuilder, sdkOperation.getAllHeaders(), HEADER);

    return methodBuilder.build();
  }

  private void addOperationParameterBinding(MethodSpec.Builder methodBuilder, List<SdkParameter> parameters,
                                            ParameterType parameterType) {
    for (SdkParameter sdkParameter : parameters) {
      if (isBoundParameter(parameterType, sdkParameter)) {
        methodBuilder.addStatement("builder.$1L($2S, $3L)",
                                   getBindingMethod(parameterType),
                                   sdkParameter.getExternalName(),
                                   sdkParameter.getJavaName());
      }
    }
  }
}
