/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.templating.sdk;

import static javax.lang.model.element.Modifier.PRIVATE;

import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.extension.api.runtime.streaming.PagingProvider;

import com.mulesoft.connectivity.rest.commons.api.connection.RestConnection;
import com.mulesoft.connectivity.rest.commons.internal.util.RestRequestBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorOperation;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.pagination.Pagination;
import com.mulesoft.connectivity.rest.sdk.templating.api.RestSdkRunConfiguration;
import com.mulesoft.connectivity.rest.sdk.templating.exception.TemplatingException;

import java.nio.file.Path;
import java.util.List;
import java.util.function.Function;

import javax.inject.Inject;

import com.squareup.javapoet.AnnotationSpec;
import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.CodeBlock;
import com.squareup.javapoet.FieldSpec;
import com.squareup.javapoet.ParameterizedTypeName;
import com.squareup.javapoet.TypeName;

public abstract class AbstractSdkPaginationOperation extends AbstractSdkOperation implements SdkPaginationStrategy {

  private final Pagination pagination;

  public AbstractSdkPaginationOperation(Path outputDir, ConnectorModel connectorModel, SdkConnector sdkConnector,
                                        ConnectorOperation operation, RestSdkRunConfiguration runConfiguration)
      throws TemplatingException {
    super(outputDir, connectorModel, sdkConnector, operation, runConfiguration);
    this.pagination = connectorModel.getPagination(operation.getPagination());
  }

  public Pagination getPagination() {
    return pagination;
  }

  @Override
  public CodeBlock getPagingMethodOperation() throws TemplatingException {
    return CodeBlock.builder()
        .addStatement(
                      "return new $T($S, $L, requestFactory, expressionLanguage, streamingHelper, $S, resolveDefaultResponseMediaType(config), overrides.getResponseTimeoutAsMillis())",
                      getPagingProviderClass(),
                      pagination.getPaginationParameter(),
                      getPagingParameter().getJavaName(),
                      getPayloadExpression())
        .build();
  }

  private SdkParameter getPagingParameter() throws TemplatingException {
    return allQueryParameters.stream().filter(x -> x.getExternalName().equalsIgnoreCase(pagination.getPaginationParameter()))
        .findFirst()
        .orElseThrow(() -> new TemplatingException("Could not get paging parameter, this is a bug."));
  }

  @Override
  public TypeName generateMethodReturn() {
    return ParameterizedTypeName.get(ClassName.get(PagingProvider.class),
                                     TypeName.get(RestConnection.class),
                                     ParameterizedTypeName.get(TypedValue.class, String.class));
  }

  @Override
  protected MessageOutputType getMessageOutputType() {
    ParameterizedTypeName parameterizedTypeName = ParameterizedTypeName.get(List.class, String.class);
    TypeName attribute = TypeName.get(Void.class);
    return new MessageOutputType(parameterizedTypeName, attribute);
  }


  @Override
  public CodeBlock generateOperationMethodBody() throws TemplatingException {
    CodeBlock.Builder paginationBody = CodeBlock.builder();

    CodeBlock.Builder requestMethodBodyBuilder = super.generateCommonOperationMethodBody();

    requestMethodBodyBuilder.addStatement("return builder");
    paginationBody.add("$T<$T,$T> requestFactory = connection -> { $L }; ",
                       Function.class,
                       RestConnection.class,
                       RestRequestBuilder.class,
                       requestMethodBodyBuilder.build());

    paginationBody.add(getPagingMethodOperation());

    return paginationBody.build();
  }

  @Override
  public FieldSpec generateExpressionLanguageField() {
    return FieldSpec
        .builder(ExpressionLanguage.class, "expressionLanguage", PRIVATE)
        .addAnnotation(AnnotationSpec.builder(Inject.class).build())
        .build();
  }

  @Override
  public String getPayloadExpression() {
    return this.pagination.getPaginationResponseExpression();
  }

  @Override
  protected boolean requiresConnectionParameter() {
    return false;
  }

  @Override
  protected boolean requiresCallbackParameter() {
    return false;
  }

  @Override
  protected boolean requiresMediaTypeAnnotation() {
    return false;
  }

}
