/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.templating.sdk.util;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.util.JavaUtils.getJavaLowerCamelNameFromXml;
import static com.mulesoft.connectivity.rest.sdk.internal.webapi.util.XmlUtils.getXmlName;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.mule.runtime.extension.api.util.NameUtils.pluralize;

import org.mule.metadata.api.model.ArrayType;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.json.api.JsonTypeLoader;

import com.mulesoft.connectivity.rest.sdk.templating.api.RestSdkRunConfiguration;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.parameter.SdkContent;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.metadata.SdkInputMetadataResolver;

import java.util.Optional;

public class ContentNameGenerator {

  private static final String CONTENT_SUFFIX = "Content";
  private static final String CONTENT_DEFAULT_NAME = "content";

  public static String getContentName(RestSdkRunConfiguration runConfiguration,
                                      SdkInputMetadataResolver metadataResolver,
                                      SdkContent.SdkContentKind contentType,
                                      String partName) {

    // Metadata was changed to be static, as 'content' was being used as name for all
    // content parameters - this produced a validation error.
    // This method generates a valid parameter name for an static metadata resolver
    // based on the type schema.
    // This method also allows using the old naming method if it is set in the runConfiguration
    // to allow maintaining backwards compatibility.

    if (runConfiguration.disambiguateContentParameters()) {
      return generateTypeSpecificContentName(metadataResolver, contentType, partName);
    } else {
      return generateSimpleContentName(contentType, partName);
    }
  }

  private static String generateSimpleContentName(SdkContent.SdkContentKind contentType, String partName) {
    return contentType.equals(SdkContent.SdkContentKind.INPUT_METADATA) ? CONTENT_DEFAULT_NAME
        : getJavaLowerCamelNameFromXml(getXmlName(partName)) + CONTENT_SUFFIX;
  }

  private static String generateTypeSpecificContentName(SdkInputMetadataResolver metadataResolver,
                                                        SdkContent.SdkContentKind contentType, String partName) {
    String schemaName = metadataResolver.getSchemaName();

    if (isNotBlank(schemaName)) {
      String typeName = null;

      if (contentType.equals(SdkContent.SdkContentKind.INPUT_METADATA)) {
        int index = metadataResolver.getSchemaName().lastIndexOf(".");

        if (index > 0) {
          schemaName = schemaName.substring(0, index);
        }

        typeName = getJavaLowerCamelNameFromXml(getXmlName(schemaName));

        // Remove generic suffixes from content parameter names.
        if (typeName.endsWith("InputSchema") && typeName.length() > 11) {
          typeName = typeName.substring(0, typeName.length() - 11);
        } else if (typeName.endsWith("OutputSchema") && typeName.length() > 12) {
          typeName = typeName.substring(0, typeName.length() - 12);
        } else if (typeName.endsWith("PartSchema")) {
          typeName = typeName.substring(0, typeName.length() - 6);
        }

        typeName = typeName + CONTENT_SUFFIX;
      } else if (contentType.equals(SdkContent.SdkContentKind.PART)) {
        typeName = partName + CONTENT_SUFFIX;
      }

      if (schemaName.endsWith(".json")) {
        final JsonTypeLoader jsonTypeLoader =
            new JsonTypeLoader(metadataResolver.getTypeDefinition().getTypeSchema().getRawSchema());
        final Optional<MetadataType> metadataType = jsonTypeLoader.load(null);

        if (metadataType.isPresent()) {
          if (metadataType.get() instanceof ArrayType) {
            return pluralize(typeName);
          }
        }
      }

      return typeName;
    }

    return generateSimpleContentName(contentType, partName);
  }
}
