/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.templating.sdk;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.util.JavaUtils.getJavaLowerCamelNameFromXml;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.util.JavaUtils.removeJavaPackageUnwantedCharacters;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.templating.VelocityTemplateEntity;
import com.mulesoft.connectivity.rest.sdk.templating.exception.TemplatingException;

import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Path;
import java.util.Properties;

import org.apache.commons.lang3.StringUtils;

public class SdkPom extends VelocityTemplateEntity {

  private final static String POM_VM = "templates/sdk/pom.vm";
  private static final String PROPERTIES_FROM_POM_PROPERTIES = "properties-from-pom.properties";
  private static final String PROJECT_VERSION = "project_version";

  private final ConnectorModel model;
  private final String restSdkCommonsVersion;
  private final String shadePackage;

  public SdkPom(Path outputDir, ConnectorModel model) throws TemplatingException {
    super(outputDir);
    this.model = model;
    this.restSdkCommonsVersion = buildProjectVersion();
    this.shadePackage = buildShadePackage();
  }

  public String getName() {
    return model.getConnectorName();
  }

  public String getDescription() {
    return model.getApiDescription();
  }

  public String getMvnGroupId() {
    return model.getMvnGroupId();
  }

  public String getMvnArtifactId() {
    return isNotBlank(model.getMvnArtifactId()) ? model.getMvnArtifactId() : model.getConnectorXmlName();
  }

  public String getMvnVersion() {
    return model.getMvnVersion();
  }

  public String getRestSdkCommonsVersion() {
    return restSdkCommonsVersion;
  }

  public String getShadePackage() {
    return shadePackage;
  }

  @Override
  public void applyTemplates() throws TemplatingException {
    applyTemplate(POM_VM, outputDir.resolve("pom.xml"));
  }

  private String buildShadePackage() {
    return removeJavaPackageUnwantedCharacters(getJavaLowerCamelNameFromXml(model.getConnectorXmlName()));
  }

  private String buildProjectVersion() throws TemplatingException {
    try (InputStream input = getClass().getClassLoader().getResourceAsStream(PROPERTIES_FROM_POM_PROPERTIES)) {
      if (input == null) {
        throw new TemplatingException("property file '" + PROPERTIES_FROM_POM_PROPERTIES
            + "' not found in the classpath, this is a bug");
      }
      Properties prop = new Properties();
      prop.load(input);
      String project_version = prop.getProperty(PROJECT_VERSION);

      if (isBlank(project_version)) {
        throw new TemplatingException("the property 'PROJECT_VERSION' (under 'PROPERTIES_FROM_POM_PROPERTIES') is either null or empty, this is a bug");
      }

      return project_version;
    } catch (IOException e) {
      throw new TemplatingException("property file '" + PROPERTIES_FROM_POM_PROPERTIES
          + "' from the classpath could not be loaded, this is a bug", e);
    }
  }
}
