/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.templating.sdk.metadata;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.util.JavaUtils.getJavaUpperCamelNameFromXml;
import static com.mulesoft.connectivity.rest.sdk.internal.webapi.util.XmlUtils.getXmlName;
import static com.mulesoft.connectivity.rest.sdk.templating.sdk.SdkConnector.API_METADATA_CATEGORY;
import static com.mulesoft.connectivity.rest.sdk.templating.sdk.resolver.SdkResolverUtil.isBoundParameterForHttpBinding;
import static com.mulesoft.connectivity.rest.sdk.templating.sdk.util.SdkTemplatingUtils.generateGetterMethod;
import static javax.lang.model.element.Modifier.PROTECTED;
import static javax.lang.model.element.Modifier.PUBLIC;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.commons.lang3.StringUtils.isBlank;

import com.mulesoft.connectivity.rest.commons.api.binding.HttpRequestBinding;
import com.mulesoft.connectivity.rest.commons.internal.model.builder.common.EvaluationContextBuilderFactory;
import com.mulesoft.connectivity.rest.commons.internal.model.common.EvaluationContext;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.generic.Argument;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.metadata.OverrideResolver;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.TypeDefinition;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.schema.TypeSchema;
import com.mulesoft.connectivity.rest.sdk.templating.JavaTemplateEntity;
import com.mulesoft.connectivity.rest.sdk.templating.api.RestSdkRunConfiguration;
import com.mulesoft.connectivity.rest.sdk.templating.exception.TemplatingException;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.SdkConnector;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.operation.AbstractSdkOperation;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.parameter.SdkParameter;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.util.FileGenerationUtil;

import java.nio.file.Path;
import java.util.List;
import java.util.Map;

import com.squareup.javapoet.CodeBlock;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.TypeSpec;

public abstract class SdkAbstractCustomFieldsMetadataResolver extends JavaTemplateEntity implements SdkMetadataResolver {


  private static final String GET_CATEGORY_NAME_METHOD_NAME = "getCategoryName";
  private static final String GET_RESOLVER_NAME_METHOD_NAME = "getResolverName";
  private static final String GET_SCHEMA_PATH_METHOD_NAME = "getSchemaPath";
  private static final String GET_METHOD_METHOD_NAME = "getMethod";
  private static final String GET_PATH_METHOD_NAME = "getPath";
  private static final String GET_TRANSFORMATION_SCRIPT_METHOD_NAME = "getTransformationScript";
  private static final String GET_SELECTOR_METHOD_NAME = "getSelector";
  private static final String GET_REQUEST_BINDINGS_METHOD_NAME = "getRequestBindings";
  private static final String ADD_URI_BINDING_METHOD = "addUriParamBinding";
  private static final String ADD_QUERY_BINDING_METHOD = "addQueryParamBinding";
  private static final String ADD_HEADER_BINDING_METHOD = "addHeaderBinding";
  private static final String BINDINGS_VAR = "bindings";

  private TypeDefinition typeDefinition;
  private TypeName typeName;
  private String internalName;
  private final String schemaName;
  private final String schemaPath;
  private final OverrideResolver overrideResolver;
  private final String selector;
  private final String className;
  private final String javaPackage;
  private final JavaTemplateEntity parentElement;
  protected Class<?> superclass;

  public SdkAbstractCustomFieldsMetadataResolver(Path outputDir, ConnectorModel connectorModel, SdkConnector sdkConnector,
                                                 String internalName, TypeDefinition typeDefinition,
                                                 FileGenerationUtil.SchemaNameType schemaNameType,
                                                 String classNameOverride, OverrideResolver overrideResolver,
                                                 String selector, JavaTemplateEntity parentElement,
                                                 RestSdkRunConfiguration runConfiguration)
      throws TemplatingException {
    super(outputDir, connectorModel, runConfiguration);

    this.typeDefinition = typeDefinition;
    typeName = sdkConnector.getTypeName();
    this.internalName = internalName;
    this.overrideResolver = overrideResolver;
    this.selector = selector;
    this.parentElement = parentElement;

    superclass = buildSuperclass();

    schemaName = processSchemaName(sdkConnector, internalName, typeDefinition, schemaNameType);

    className = buildClassName(internalName, classNameOverride);
    javaPackage = buildPackage(connectorModel);

    Path schemaOutputDir = sdkConnector.getResourcesPath().resolve(runConfiguration.getGeneratedSchemasDir());
    FileGenerationUtil.writeSchema(typeDefinition.getTypeSchema(),
                                   schemaOutputDir,
                                   schemaName);

    schemaPath = "/" + runConfiguration.getGeneratedSchemasDir() + "/" + schemaName;

  }

  protected abstract Class<?> buildSuperclass() throws TemplatingException;

  protected abstract String getClassNameSuffix();

  protected abstract String buildGetResolverName(String internalName);

  private String processSchemaName(SdkConnector sdkConnector, String internalName,
                                   TypeDefinition typeDefinition, FileGenerationUtil.SchemaNameType schemaNameType) {
    Map<TypeSchema, String> typeSchemaNames = sdkConnector.getTypeSchemaNames();
    TypeSchema source = typeDefinition.getTypeSchema();
    return typeSchemaNames
        .computeIfAbsent(source,
                         typeSchema -> FileGenerationUtil.generateSchemaName(source, internalName, schemaNameType, EMPTY));
  }

  private String buildClassName(String internalName, String classNameOverride) {
    if (isBlank(classNameOverride)) {
      return getJavaUpperCamelNameFromXml(internalName)
          + getClassNameSuffix();
    }

    return getJavaUpperCamelNameFromXml(getXmlName(classNameOverride))
        + getClassNameSuffix();
  }

  private String buildPackage(ConnectorModel connectorModel) {
    return connectorModel.getBasePackage() + ".internal.metadata";
  }

  @Override
  public void applyTemplates() throws TemplatingException {
    generateMetadataResolverClass();
  }

  @Override
  public String getClassName() {
    return className;
  }

  @Override
  public String getPackage() {
    return javaPackage;
  }

  @Override
  public boolean getRequiresMetadataResolver() {
    return true;
  }

  @Override
  public TypeDefinition getTypeDefinition() {
    return typeDefinition;
  }

  @Override
  public String getSchemaName() {
    return schemaName;
  }

  private void generateMetadataResolverClass() throws TemplatingException {
    TypeSpec.Builder typeResolverClassBuilder =
        TypeSpec
            .classBuilder(className)
            .addModifiers(PUBLIC)
            .superclass(superclass);

    generateMethods(typeResolverClassBuilder);

    writeClassToFile(typeResolverClassBuilder.build(), javaPackage);
  }

  protected void generateMethods(TypeSpec.Builder typeResolverClassBuilder) {
    generateGetCategoryNameMethod(typeResolverClassBuilder);
    generateGetterMethod(typeResolverClassBuilder, GET_RESOLVER_NAME_METHOD_NAME, buildGetResolverName(internalName));
    generateGetterMethod(typeResolverClassBuilder, GET_SCHEMA_PATH_METHOD_NAME, schemaPath);
    generateGetterMethod(typeResolverClassBuilder, GET_METHOD_METHOD_NAME, overrideResolver.getRequest().getMethod().toString());
    generateGetterMethod(typeResolverClassBuilder, GET_PATH_METHOD_NAME, overrideResolver.getRequest().getPath());
    generateGetterMethod(typeResolverClassBuilder, GET_TRANSFORMATION_SCRIPT_METHOD_NAME,
                         overrideResolver.getTransformationScript());
    generateGetterMethod(typeResolverClassBuilder, GET_SELECTOR_METHOD_NAME, selector);
    if (overrideResolver.getRequest().getHttpRequestBinding() != null) {
      generateGetRequestBindingsMethod(typeResolverClassBuilder, overrideResolver.getRequest().getHttpRequestBinding());
      generateBuildEvaluationContextMethod(typeResolverClassBuilder, overrideResolver.getRequest().getHttpRequestBinding());
    }
  }

  private void generateGetCategoryNameMethod(TypeSpec.Builder typeResolverClassBuilder) {
    CodeBlock getCategoryCodeBlock = CodeBlock.builder()
        .addStatement("return $T." + API_METADATA_CATEGORY, typeName)
        .build();

    MethodSpec getCategoryNameMethod = MethodSpec.methodBuilder(GET_CATEGORY_NAME_METHOD_NAME)
        .returns(TypeName.get(String.class))
        .addModifiers(PUBLIC)
        .addAnnotation(Override.class)
        .addCode(getCategoryCodeBlock)
        .build();

    typeResolverClassBuilder.addMethod(getCategoryNameMethod);
  }

  private void generateGetRequestBindingsMethod(TypeSpec.Builder typeResolverClassBuilder,
                                                com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dataexpression.httprequest.HttpRequestBinding httpRequestBinding) {

    MethodSpec.Builder methodBuilder =
        MethodSpec.methodBuilder(GET_REQUEST_BINDINGS_METHOD_NAME)
            .addModifiers(PROTECTED)
            .returns(TypeName.get(HttpRequestBinding.class))
            .addAnnotation(Override.class);

    methodBuilder.addStatement("$1T $2L = new $1T()",
                               HttpRequestBinding.class,
                               BINDINGS_VAR);

    for (Argument binding : httpRequestBinding.getQueryParameter()) {
      addBinding(methodBuilder, binding, ADD_QUERY_BINDING_METHOD);
    }
    for (Argument binding : httpRequestBinding.getUriParameter()) {
      addBinding(methodBuilder, binding, ADD_URI_BINDING_METHOD);
    }
    for (Argument binding : httpRequestBinding.getHeader()) {
      addBinding(methodBuilder, binding, ADD_HEADER_BINDING_METHOD);
    }

    methodBuilder.addStatement("return $1L", BINDINGS_VAR);
    typeResolverClassBuilder.addMethod(methodBuilder.build());

  }

  private void addBinding(MethodSpec.Builder methodBuilder, Argument binding, String bindingMethod) {
    methodBuilder.addStatement("$L.$L($S, $S)",
                               BINDINGS_VAR,
                               bindingMethod,
                               binding.getName(),
                               binding.getValue().getValue());
  }

  private void generateBuildEvaluationContextMethod(TypeSpec.Builder typeResolverClassBuilder,
                                                    com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dataexpression.httprequest.HttpRequestBinding httpRequestBinding) {
    MethodSpec.Builder methodBuilder =
        MethodSpec.methodBuilder("buildEvaluationContext")
            .addModifiers(PROTECTED)
            .returns(EvaluationContext.class)
            .addParameter(TypeName.get(EvaluationContextBuilderFactory.class), "builderFactory")
            .addParameter(TypeName.get(Object.class), "key")
            .addAnnotation(Override.class);

    addParameterBindings(methodBuilder, ((AbstractSdkOperation) parentElement).getAllParameters(), httpRequestBinding);

    typeResolverClassBuilder.addMethod(methodBuilder.build());
  }

  private void addParameterBindings(MethodSpec.Builder methodBuilder, List<SdkParameter> parameters,
                                    com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dataexpression.httprequest.HttpRequestBinding httpRequestBinding) {

    CodeBlock.Builder contextBindingBuilder = CodeBlock.builder();

    contextBindingBuilder.add("return builderFactory.operationContextBuilder()");
    for (SdkParameter sdkParameter : parameters) {
      if (isBoundParameterForHttpBinding(sdkParameter, httpRequestBinding)) {
        contextBindingBuilder.add(".$1L($2S, $3L)",
                                  sdkParameter.getParameterType().getAccessorName(),
                                  sdkParameter.getExternalName(),
                                  "key");
      }
    }

    contextBindingBuilder.add(".build()");
    methodBuilder.addStatement(contextBindingBuilder.build());
  }

}
