/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.templating.sdk.metadata;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.util.JavaUtils.getJavaUpperCamelNameFromXml;
import static com.mulesoft.connectivity.rest.sdk.internal.webapi.util.XmlUtils.getXmlName;
import static com.mulesoft.connectivity.rest.sdk.templating.sdk.SdkConnector.API_METADATA_CATEGORY;
import static com.mulesoft.connectivity.rest.sdk.templating.sdk.util.FileGenerationUtil.supportsSchemaGeneration;
import static javax.lang.model.element.Modifier.PUBLIC;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

import org.mule.metadata.api.model.MetadataFormat;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.EmptyTypeDefinition;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.TypeDefinition;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.schema.TypeSchema;
import com.mulesoft.connectivity.rest.sdk.templating.JavaTemplateEntity;
import com.mulesoft.connectivity.rest.sdk.templating.api.RestSdkRunConfiguration;
import com.mulesoft.connectivity.rest.sdk.templating.exception.TemplatingException;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.SdkConnector;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.util.FileGenerationUtil;
import com.squareup.javapoet.CodeBlock;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.TypeSpec;

import java.nio.file.Path;
import java.util.Map;

public abstract class SdkAbstractStaticMetadataResolver extends JavaTemplateEntity implements SdkMetadataResolver {

  private final TypeDefinition typeDefinition;

  private static final String GET_SCHEMA_PATH_METHOD_NAME = "getSchemaPath";
  private static final String GET_RESOLVER_NAME_METHOD_NAME = "getResolverName";
  private static final String GET_CATEGORY_NAME_METHOD_NAME = "getCategoryName";
  private static final String GET_QNAME_METHOD_NAME = "getQName";
  private static final String GET_FORMAT_METHOD_NAME = "getFormat";

  private final boolean requiresMetadataResolver;
  private final boolean requiresToWriteSchema;

  private String schemaName;
  private String schemaPath;

  private String className;
  private String javaPackage;
  private CodeBlock getResolverNameCodeBlock;
  private CodeBlock getCategoryCodeBlock;
  protected Class<?> superclass;

  /***
   * Creates a type resolver that will generate a schema file in the resources directory if needed.
   * 
   * @param sdkConnector the parent SdkConnector for this type
   * @param typeDefinition The type definition which this metadata will be created for
   * @param internalName The internal name of the container. It will be used to generate the schema name.
   * @param schemaNameType The type of name that this schema will get.
   * @param partName When this schema corresponds to a multipart part, this part name will be used to generate the schema name.
   *        Must be an empty string if it he type definition does not correspond to a part.
   * @param classNameOverride When specified forces the class name to be generated based on this instead of using the operation
   *        and part names.
   * @param runConfiguration The configuration for this execution of the connector generation.
   */
  public SdkAbstractStaticMetadataResolver(Path outputDir, ConnectorModel connectorModel, SdkConnector sdkConnector,
                                           String internalName, TypeDefinition typeDefinition,
                                           FileGenerationUtil.SchemaNameType schemaNameType,
                                           String partName, String classNameOverride, RestSdkRunConfiguration runConfiguration)
      throws TemplatingException {
    super(outputDir, connectorModel, runConfiguration);

    this.typeDefinition = typeDefinition;

    superclass = buildSuperclass(typeDefinition);
    requiresMetadataResolver = buildRequiresMetadataResolver();
    requiresToWriteSchema = requiresToWriteSchema();

    if (supportsSchemaGeneration(typeDefinition.getTypeSchema())) {
      schemaName = processSchemaName(sdkConnector, internalName, typeDefinition, schemaNameType, partName);

      if (requiresMetadataResolver) {
        className = buildClassName(internalName, partName, classNameOverride);
        javaPackage = buildPackage(connectorModel);
        getCategoryCodeBlock = CodeBlock.builder()
            .addStatement("return $T." + API_METADATA_CATEGORY, sdkConnector.getTypeName())
            .build();
        getResolverNameCodeBlock = CodeBlock.builder()
            .addStatement("return $S", buildGetResolverName(internalName))
            .build();

        if (requiresToWriteSchema) {
          Path schemaOutputDir = sdkConnector.getResourcesPath().resolve(runConfiguration.getGeneratedSchemasDir());
          FileGenerationUtil.writeSchema(typeDefinition.getTypeSchema(),
                                         schemaOutputDir,
                                         schemaName);

          schemaPath = "/" + runConfiguration.getGeneratedSchemasDir() + "/" + schemaName;
        }
      }
    }
  }

  protected abstract String buildGetResolverName(String internalName);

  private String processSchemaName(SdkConnector sdkConnector, String internalName,
                                   TypeDefinition typeDefinition, FileGenerationUtil.SchemaNameType schemaNameType,
                                   String partName) {
    Map<TypeSchema, String> typeSchemaNames = sdkConnector.getTypeSchemaNames();
    TypeSchema source = typeDefinition.getTypeSchema();
    return typeSchemaNames
        .computeIfAbsent(source, typeSchema -> getSchemaName(internalName, schemaNameType, source, partName));
  }

  protected String getSchemaName(String internalName, FileGenerationUtil.SchemaNameType schemaNameType,
                                 TypeSchema source,
                                 String partName) {
    return FileGenerationUtil.generateSchemaName(source,
                                                 internalName,
                                                 schemaNameType,
                                                 partName);
  }

  protected abstract Class<?> buildSuperclass(TypeDefinition typeDefinition) throws TemplatingException;

  protected abstract String getClassNameSuffix();

  private String buildClassName(String internalName, String partName, String classNameOverride) {
    if (isBlank(classNameOverride)) {
      return getJavaUpperCamelNameFromXml(internalName)
          + (isNotBlank(partName) ? getJavaUpperCamelNameFromXml(getXmlName(partName)) : EMPTY)
          + getClassNameSuffix();
    }

    return getJavaUpperCamelNameFromXml(getXmlName(classNameOverride))
        + getClassNameSuffix();
  }

  private String buildPackage(ConnectorModel connectorModel) {
    return connectorModel.getBasePackage() + ".internal.metadata";
  }

  public String getClassName() {
    return className;
  }

  public String getPackage() {
    return javaPackage;
  }

  @Override
  public void applyTemplates() throws TemplatingException {
    if (requiresMetadataResolver) {
      generateMetadataResolverClass();
    }
  }

  private void generateMetadataResolverClass() throws TemplatingException {
    TypeSpec.Builder typeResolverClassBuilder =
        TypeSpec
            .classBuilder(className)
            .addModifiers(PUBLIC)
            .superclass(superclass);

    generateMethods(typeResolverClassBuilder);

    writeClassToFile(typeResolverClassBuilder.build(), javaPackage);
  }

  protected void generateMethods(TypeSpec.Builder typeResolverClassBuilder) {
    if (requiresToWriteSchema) {
      addGetSchemaPathMethod(typeResolverClassBuilder);
    }

    addGetResolverNameMethod(typeResolverClassBuilder);

    addGetCategoryNameMethod(typeResolverClassBuilder);

    if (requiresQNameMethod()) {
      addGetQNameMethod(typeResolverClassBuilder);
    }

    if (requiresFormatMethod()) {
      addFormatMethod(typeResolverClassBuilder);
    }
  }

  private void addFormatMethod(TypeSpec.Builder typeResolverClassBuilder) {
    CodeBlock methodBody = CodeBlock.builder()
        .addStatement("return new $1T($2S, $2S, $2S)",
                      MetadataFormat.class,
                      typeDefinition.getMediaType().getType() + "/" + typeDefinition.getMediaType().getSubtype())
        .build();

    MethodSpec formatMethod = MethodSpec.methodBuilder(GET_FORMAT_METHOD_NAME)
        .returns(TypeName.get(MetadataFormat.class))
        .addModifiers(PUBLIC)
        .addAnnotation(Override.class)
        .addCode(methodBody)
        .build();

    typeResolverClassBuilder.addMethod(formatMethod);
  }

  private void addGetSchemaPathMethod(TypeSpec.Builder typeResolverClassBuilder) {
    CodeBlock methodBody = CodeBlock.builder()
        .addStatement("return $S", schemaPath)
        .build();

    MethodSpec createConnectionMethod = MethodSpec.methodBuilder(GET_SCHEMA_PATH_METHOD_NAME)
        .returns(TypeName.get(String.class))
        .addModifiers(PUBLIC)
        .addAnnotation(Override.class)
        .addCode(methodBody)
        .build();

    typeResolverClassBuilder.addMethod(createConnectionMethod);
  }

  private void addGetResolverNameMethod(TypeSpec.Builder typeResolverClassBuilder) {
    MethodSpec getCategoryNameMethod = MethodSpec.methodBuilder(GET_RESOLVER_NAME_METHOD_NAME)
        .returns(TypeName.get(String.class))
        .addModifiers(PUBLIC)
        .addAnnotation(Override.class)
        .addCode(getResolverNameCodeBlock)
        .build();

    typeResolverClassBuilder.addMethod(getCategoryNameMethod);
  }


  private void addGetCategoryNameMethod(TypeSpec.Builder typeResolverClassBuilder) {
    MethodSpec getCategoryNameMethod = MethodSpec.methodBuilder(GET_CATEGORY_NAME_METHOD_NAME)
        .returns(TypeName.get(String.class))
        .addModifiers(PUBLIC)
        .addAnnotation(Override.class)
        .addCode(getCategoryCodeBlock)
        .build();

    typeResolverClassBuilder.addMethod(getCategoryNameMethod);
  }

  private void addGetQNameMethod(TypeSpec.Builder typeResolverClassBuilder) {
    CodeBlock methodBody = CodeBlock.builder()
        .addStatement("return $S", getQName())
        .build();

    MethodSpec getCategoryNameMethod = MethodSpec.methodBuilder(GET_QNAME_METHOD_NAME)
        .returns(TypeName.get(String.class))
        .addModifiers(PUBLIC)
        .addAnnotation(Override.class)
        .addCode(methodBody)
        .build();

    typeResolverClassBuilder.addMethod(getCategoryNameMethod);
  }

  protected abstract boolean requiresQNameMethod();

  protected abstract boolean requiresFormatMethod();

  protected abstract String getQName();

  protected boolean buildRequiresMetadataResolver() {
    return !(typeDefinition instanceof EmptyTypeDefinition);
  }

  public boolean getRequiresMetadataResolver() {
    return requiresMetadataResolver;
  }

  protected abstract boolean requiresToWriteSchema();

  public String getSchemaName() {
    return schemaName;
  }

  public TypeDefinition getTypeDefinition() {
    return typeDefinition;
  }
}
