/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.templating.sdk.resolver;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dw.DataWeaveExpressionParser.selectionsFromBinding;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.util.JavaUtils.getParameterJavaName;
import static com.mulesoft.connectivity.rest.sdk.internal.webapi.util.XmlUtils.getXmlName;

import com.mulesoft.connectivity.rest.commons.api.dw.DWBindings;
import com.mulesoft.connectivity.rest.commons.api.dw.HttpRequestDWBinding;
import com.mulesoft.connectivity.rest.commons.internal.model.dataexpressions.BindingField;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dataexpression.httprequest.HttpRequestBinding;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.expression.Expression;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.generic.Argument;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.parameter.SdkParameter;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class SdkResolverUtil {

  public static String getBindingMethod(ParameterType parameterType) {
    switch (parameterType) {
      case URI:
      case QUERY:
      case HEADER:
        return parameterType.getAccessorName();
      default:
        throw new IllegalArgumentException("Parameter type not supported. This is a bug.");
    }
  }

  public static boolean isBoundParameterForHttpBinding(SdkParameter sdkParameter,
                                                       HttpRequestBinding httpBinding) {
    if (httpBinding == null) {
      return false;
    }

    List<Expression> expressions;

    switch (sdkParameter.getParameterType()) {
      case URI:
      case QUERY:
      case HEADER:
      case AUXILIAR:
      case TRIGGER:
        expressions = httpBinding.getAllExpressions();
        break;
      default:
        throw new IllegalArgumentException("Parameter type '" + sdkParameter.getParameterType()
            + "' not supported. This is a bug.");
    }

    return expressions.stream().anyMatch(x -> isBoundScript(sdkParameter, x.getValue()));
  }

  public static boolean isBoundScript(SdkParameter sdkParameter, String script) {
    if (script == null) {
      return false;
    }
    String bindingName = sdkParameter.getParameterType().getBinding();
    String[] accessors = selectionsFromBinding(script, bindingName);
    accessors = lookForHttpBindings(sdkParameter, script, accessors);
    return Arrays.asList(accessors).contains(sdkParameter.getExternalName());
  }

  /**
   * Looks for special bindings in cases where the current parameter is of type query, uri or header, if so then it assumes the
   * binding could be of an HTTP one, which relies in the object {@link HttpRequestDWBinding}
   *
   * @param sdkParameter to look of which type is it
   * @param script to analyze whether it has the http binding or not
   * @param accessors list of accessors to contribute if binding was found
   * @return the array of accessors's api name (real API name, and not the rest-sdk generated one) to look for if the parameter is
   *         being used
   */
  private static String[] lookForHttpBindings(SdkParameter sdkParameter, String script, String[] accessors) {
    String httpNamespaceSelector = null;
    switch (sdkParameter.getParameterType()) {
      case URI:
        httpNamespaceSelector = DWBindings.PATH.getBinding();
        break;
      case QUERY:
        httpNamespaceSelector = DWBindings.QUERY.getBinding();
        break;
      case HEADER:
        httpNamespaceSelector = DWBindings.HEADER.getBinding();
        break;
      default:
        break;
    }
    if (httpNamespaceSelector != null) {
      String[] httpBindingAccessors =
          selectionsFromBinding(script, DWBindings.REQUEST.getBinding(), Arrays.asList(httpNamespaceSelector));
      accessors = Stream.concat(Arrays.stream(accessors), Arrays.stream(httpBindingAccessors))
          .toArray(String[]::new);
    }
    return accessors;
  }

  public static List<BindingField> getBodyFieldsFromHttpBinding(HttpRequestBinding httpBinding) {
    if (httpBinding == null) {
      return Collections.emptyList();
    }

    final String searchParam = ParameterType.BODY.getBinding() + ".";

    List<Argument> parameterBindings = new ArrayList<>();
    parameterBindings.addAll(httpBinding.getUriParameter());
    parameterBindings.addAll(httpBinding.getQueryParameter());
    parameterBindings.addAll(httpBinding.getHeader());

    return parameterBindings.stream()
        .filter(x -> x.getValue().getValue().contains(searchParam))
        .map(argument -> new BindingField(argument.getName(), argument.getValue().getValue().replace(searchParam, "")))
        .collect(Collectors.toList());
  }

  public static String getActingParameterJavaName(Argument argument) {
    return getParameterJavaName(
                                getXmlName(argument.getName()),
                                false);
  }

}
