/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.templating;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.util.JavaUtils.getJavaUpperCamelNameFromXml;
import static java.lang.Integer.MAX_VALUE;
import static java.lang.String.format;
import static java.nio.charset.StandardCharsets.UTF_8;
import static java.nio.file.Files.copy;
import static java.nio.file.Files.delete;
import static java.nio.file.Files.find;
import static java.nio.file.Files.readAllBytes;
import static java.nio.file.Files.write;
import static java.nio.file.StandardCopyOption.REPLACE_EXISTING;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.logging.log4j.LogManager.getLogger;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.templating.api.RestSdkRunConfiguration;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.SdkConnector;
import com.mulesoft.connectivity.rest.sdk.templating.exception.TemplatingException;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.SdkGitIgnore;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.SdkLicense;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.SdkLicenseHeader;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.SdkPom;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.munit.SdkTestLog4j;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Path;
import java.util.List;
import java.util.stream.Stream;

import com.google.googlejavaformat.java.Formatter;

import org.apache.logging.log4j.Logger;

public class SdkConnectorTemplateEntity extends TemplateEntity {

  private static final Logger LOGGER = getLogger(SdkConnectorTemplateEntity.class);

  private static final Formatter FORMATTER = new Formatter();
  private final SdkConnector sdkConnector;
  private final SdkPom sdkPom;
  private final SdkTestLog4j sdkTestLog4j;
  private final Path projectDir;
  private final List<File> descriptors;
  private final SdkGitIgnore sdkGitIgnore;
  private final SdkLicense sdkLicense;
  private final SdkLicenseHeader sdkLicenseHeader;
  private final RestSdkRunConfiguration runConfiguration;

  public SdkConnectorTemplateEntity(ConnectorModel connectorModel, List<File> descriptors,
                                    RestSdkRunConfiguration runConfiguration, Path outputDir)
      throws TemplatingException {
    String projectName = getJavaUpperCamelNameFromXml(connectorModel.getConnectorXmlName());
    projectDir = outputDir.resolve(projectName);
    checkValidProjectDir(projectDir);

    this.sdkConnector = new SdkConnector(projectDir, connectorModel, runConfiguration);
    this.sdkPom = new SdkPom(projectDir, connectorModel);
    this.descriptors = descriptors;
    this.sdkGitIgnore = new SdkGitIgnore(projectDir);
    this.sdkLicense = new SdkLicense(projectDir, connectorModel);
    this.sdkLicenseHeader = new SdkLicenseHeader(projectDir, connectorModel);
    this.sdkTestLog4j = new SdkTestLog4j(projectDir);
    this.runConfiguration = runConfiguration;
  }

  private void checkValidProjectDir(Path directory) throws TemplatingException {
    final File dirFile = directory.toFile();
    if (dirFile.exists() && dirFile.isDirectory()) {
      final String[] children = dirFile.list();
      if (children != null && children.length > 0) {
        throw new TemplatingException(format("Output directory is not empty: %s", projectDir.toString()));
      }
    }
    LOGGER.info("Project output directory: " + projectDir.toString());
  }

  @Override
  public void applyTemplates() throws TemplatingException {
    sdkConnector.applyTemplates();
    sdkPom.applyTemplates();
    sdkGitIgnore.applyTemplates();
    sdkLicense.applyTemplates();
    sdkLicenseHeader.applyTemplates();
    sdkTestLog4j.applyTemplates();
    formatJavaFiles();
    addDescriptorToSources();
  }

  protected void addDescriptorToSources() throws TemplatingException {
    if (descriptors != null && !descriptors.isEmpty()) {
      int i = 1;
      for (File descriptor : descriptors) {
        try (InputStream descriptorStream = new FileInputStream(descriptor)) {
          String targetFilename = format("connector-descriptor%s.yaml", i > 1 ? ("-" + i) : EMPTY);
          copy(descriptorStream, sdkConnector.getSourcesPath().resolve(targetFilename), REPLACE_EXISTING);
          i++;
        } catch (IOException e) {
          throw new TemplatingException("Could not add descriptor to generated sources");
        }
      }
    }
  }

  private void formatJavaFiles() throws TemplatingException {
    for (Path path : getJavaFiles()) {
      javaFormat(path);
    }
  }

  private List<Path> getJavaFiles() throws TemplatingException {
    try (Stream<Path> paths = find(projectDir,
                                   MAX_VALUE,
                                   (filePath, fileAttr) -> filePath.toString().toLowerCase().endsWith(".java"))) {
      return paths.collect(toList());
    } catch (Exception e) {
      throw new TemplatingException("Could not find files to format", e);
    }
  }

  private void javaFormat(Path filepath) throws TemplatingException {
    String formattedSource;

    try {
      String content = new String(readAllBytes(filepath), UTF_8);
      formattedSource = FORMATTER.formatSource(content);

      delete(filepath);
      write(filepath, formattedSource.getBytes(UTF_8));
    } catch (Exception e) {
      throw new TemplatingException("Could not format file " + filepath.toString(), e);
    }
  }
}
