/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.templating.sdk;

import static java.util.stream.Collectors.toList;
import static javax.lang.model.element.Modifier.PUBLIC;

import org.mule.runtime.extension.api.annotation.Configuration;
import org.mule.runtime.extension.api.annotation.Operations;
import org.mule.runtime.extension.api.annotation.Sources;
import org.mule.runtime.extension.api.annotation.connectivity.ConnectionProviders;

import com.mulesoft.connectivity.rest.commons.api.configuration.RestConfiguration;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorOperation;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.ConnectorSecurityScheme;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.trigger.Trigger;
import com.mulesoft.connectivity.rest.sdk.templating.JavaTemplateEntity;
import com.mulesoft.connectivity.rest.sdk.templating.api.RestSdkRunConfiguration;
import com.mulesoft.connectivity.rest.sdk.templating.exception.TemplatingException;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.operation.AbstractSdkOperation;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.operation.SdkOperationFactory;

import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;

import com.squareup.javapoet.AnnotationSpec;
import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.CodeBlock;
import com.squareup.javapoet.TypeSpec;

public class SdkConfig extends JavaTemplateEntity {

  private final SdkConnector connector;
  private final List<SdkConnectionProvider> connectionProviders;
  private final List<AbstractSdkOperation> operations;
  private final List<SdkTrigger> triggers;

  public SdkConfig(Path outputDir,
                   ConnectorModel connectorModel,
                   SdkConnector sdkConnector,
                   RestSdkRunConfiguration runConfiguration)
      throws TemplatingException {
    super(outputDir, connectorModel, runConfiguration);
    this.connector = sdkConnector;

    this.connectionProviders = buildConnectionProviders(outputDir, connectorModel, sdkConnector);

    this.operations = new ArrayList<>();
    for (ConnectorOperation operation : connectorModel.getOperations()) {
      this.operations.add(SdkOperationFactory.create(outputDir, connectorModel, sdkConnector, operation, runConfiguration));
    }

    this.triggers = new ArrayList<>();
    for (Trigger triggerModel : connectorModel.getTriggers()) {
      triggers.add(new SdkTrigger(outputDir, connectorModel, sdkConnector, triggerModel, runConfiguration));
    }
  }

  private List<SdkConnectionProvider> buildConnectionProviders(Path outputDir, ConnectorModel connectorModel,
                                                               SdkConnector sdkConnector)
      throws TemplatingException {
    List<ConnectorSecurityScheme> securitySchemeModels = connectorModel.getOperations().stream()
        .map(ConnectorOperation::getSecuritySchemes)
        .flatMap(List::stream)
        .distinct()
        .collect(toList());

    final List<SdkConnectionProvider> list = new ArrayList<>();
    for (ConnectorSecurityScheme scheme : securitySchemeModels) {
      list.add(new SdkConnectionProvider(outputDir, connectorModel, sdkConnector, scheme, runConfiguration));
    }
    return list;
  }

  public String getJavaClassName() {
    String baseName =
        connector.getJavaName().endsWith("Connector") ? connector.getJavaName().substring(0, connector.getJavaName().length() - 9)
            : connector.getJavaName();

    return baseName + "Configuration";
  }

  public String getPackage() {
    return connectorModel.getBasePackage() + ".internal.config";
  }

  @Override
  public void applyTemplates() throws TemplatingException {
    generateConfigClass();

    for (SdkConnectionProvider connectionProvider : connectionProviders) {
      connectionProvider.applyTemplates();
    }

    for (AbstractSdkOperation operation : operations) {
      operation.applyTemplates();
    }

    for (SdkTrigger trigger : triggers) {
      trigger.applyTemplates();
    }
  }

  private void generateConfigClass() throws TemplatingException {
    TypeSpec.Builder configClassBuilder =
        TypeSpec
            .classBuilder(getJavaClassName())
            .addModifiers(PUBLIC)
            .superclass(RestConfiguration.class)
            .addAnnotation(getConfigurationAnnotation())
            .addAnnotation(getOperationsAnnotation())
            .addAnnotation(getConnectionProvidersAnnotation());

    if (!triggers.isEmpty()) {
      configClassBuilder.addAnnotation(getSourcesAnnotation());
    }

    writeClassToFile(configClassBuilder.build(), getPackage());
  }

  private AnnotationSpec getSourcesAnnotation() {
    CodeBlock.Builder codeBlock = CodeBlock.builder();

    codeBlock.add("{");

    for (int i = 0; i < triggers.size(); i++) {
      SdkTrigger trigger = triggers.get(i);
      codeBlock.add("$T.class", ClassName.get(trigger.getPackage(), trigger.getJavaClassName()));
      if (i < triggers.size() - 1) {
        codeBlock.add(",");
      }
    }

    codeBlock.add("}");

    AnnotationSpec.Builder sourcesAnnotationBuilder =
        AnnotationSpec
            .builder(Sources.class)
            .addMember(VALUE_MEMBER, codeBlock.build());

    return sourcesAnnotationBuilder.build();
  }

  private AnnotationSpec getOperationsAnnotation() {
    CodeBlock.Builder codeBlock = CodeBlock.builder();

    codeBlock.add("{");

    for (int i = 0; i < operations.size(); i++) {
      AbstractSdkOperation operation = operations.get(i);
      codeBlock.add("$T.class", ClassName.get(operation.getPackage(), operation.getJavaClassName()));
      if (i < operations.size() - 1) {
        codeBlock.add(",");
      }
    }

    codeBlock.add("}");

    AnnotationSpec.Builder operationsAnnotationBuilder =
        AnnotationSpec
            .builder(Operations.class)
            .addMember(VALUE_MEMBER, codeBlock.build());

    return operationsAnnotationBuilder.build();
  }

  private AnnotationSpec getConnectionProvidersAnnotation() {
    CodeBlock.Builder codeBlock = CodeBlock.builder();

    codeBlock.add("{");

    for (int i = 0; i < connectionProviders.size(); i++) {
      SdkConnectionProvider connectionProvider = connectionProviders.get(i);
      codeBlock.add("$T.class", ClassName.get(connectionProvider.getPackage(), connectionProvider.getJavaClassName()));
      if (i < connectionProviders.size() - 1) {
        codeBlock.add(",");
      }
    }

    codeBlock.add("}");

    AnnotationSpec.Builder connectionProvidersAnnotationBuilder =
        AnnotationSpec
            .builder(ConnectionProviders.class)
            .addMember(VALUE_MEMBER, codeBlock.build());

    return connectionProvidersAnnotationBuilder.build();
  }

  private AnnotationSpec getConfigurationAnnotation() {
    AnnotationSpec.Builder extensionAnnotationBuilder =
        AnnotationSpec
            .builder(Configuration.class);

    return extensionAnnotationBuilder.build();
  }

  public List<AbstractSdkOperation> getOperations() {
    return this.operations;
  }
}
