/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.templating.sdk.resolver;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.generic.Argument;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.resolver.ResolverDeclaration;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.resolver.ResolverReference;
import com.mulesoft.connectivity.rest.sdk.templating.api.RestSdkRunConfiguration;
import com.mulesoft.connectivity.rest.sdk.templating.exception.TemplatingException;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.parameter.SdkParameter;

import com.squareup.javapoet.CodeBlock;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.TypeSpec;

import java.nio.file.Path;
import java.util.Map;

public abstract class AbstractSdkResolverProviderReference extends AbstractSdkResolverProvider {

  public static final String RESOLVER_NAME_FIELD = "RESOLVER_NAME";

  private final ResolverReference<?> reference;
  private final SdkResolverDefinition<?> sdkResolverDefinition;
  private final ResolverDeclaration<?> declaration;
  private TypeName superclass;

  public AbstractSdkResolverProviderReference(Path outputDir,
                                              ConnectorModel connectorModel,
                                              ResolverReference<?> reference,
                                              SdkResolverDefinition<?> sdkResolverDefinition,
                                              RestSdkRunConfiguration runConfiguration) {
    super(outputDir, connectorModel, runConfiguration);
    this.reference = reference;
    this.sdkResolverDefinition = sdkResolverDefinition;
    this.declaration = reference.getDeclaration();
  }

  /**
   * Generates the 'build' method of the resolver reference.
   */
  protected void addBuildMethod(TypeSpec.Builder classBuilder) {
    sdkResolverDefinition.addBuildMethod(classBuilder, generateBuildMethodBody());
  }

  private CodeBlock generateBuildMethodBody() {
    CodeBlock.Builder builder = CodeBlock.builder();

    builder.add("builder.reference(referenceBuilder -> referenceBuilder.declaration($1L)", RESOLVER_NAME_FIELD);

    for (Argument arg : reference.getArguments()) {
      builder.add(".argument($1S, argumentBuilder -> argumentBuilder.value(valueBuilder -> valueBuilder.expression($2S)))",
                  arg.getName(),
                  arg.getValue().getValue());
    }
    builder.add(")");

    return builder.build();
  }

  @Override
  public void applyTemplates() throws TemplatingException {
    if (getAllParents().containsKey(declaration.getName())) {
      this.superclass = getAllParents().get(declaration.getName());
    } else {
      AbstractSdkResolverProviderReferenceParent parent = getNewParent();
      parent.applyTemplates();
      this.superclass = parent.getTypeName();
      getAllParents().put(declaration.getName(), superclass);
    }

    super.applyTemplates();
  }

  protected abstract Map<String, TypeName> getAllParents();

  protected abstract AbstractSdkResolverProviderReferenceParent getNewParent();

  @Override
  protected TypeName getSuperClass() {
    return superclass;
  }

  @Override
  protected boolean isBoundParameter(SdkParameter sdkParameter) {
    final String searchParam = sdkParameter.getParameterType().getAccessorName() + "." + sdkParameter.getExternalName();

    return reference.getArguments().stream()
        .anyMatch(x -> x.getValue().getValue().contains(searchParam));
  }

  @Override
  protected void buildClass(TypeSpec.Builder classBuilder) {
    addParameterFieldsIfNeeded(classBuilder);
    addBuildMethod(classBuilder);
    addConfigureEvaluationContextMethod(classBuilder);
  }
}
