/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.templating;

import static com.google.common.base.CaseFormat.UPPER_CAMEL;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dw.DataWeaveExpressionParser.isBindingUsed;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dw.DataWeaveExpressionParser.isBodyBindingUsed;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.BODY;
import static com.mulesoft.connectivity.rest.sdk.templating.sdk.resolver.SdkResolverUtil.getActingParameterJavaName;
import static com.mulesoft.connectivity.rest.sdk.templating.sdk.util.SdkBodyLevelUtils.transformBodyPrefix;
import static java.util.Collections.emptyList;
import static javax.lang.model.element.Modifier.FINAL;
import static javax.lang.model.element.Modifier.PRIVATE;
import static javax.lang.model.element.Modifier.PUBLIC;
import static javax.lang.model.element.Modifier.STATIC;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dw.DataWeaveExpressionParser;
import org.mule.sdk.api.annotation.binding.Binding;
import org.mule.sdk.api.annotation.values.FieldValues;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dataexpression.httprequest.HttpRequestBinding;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.generic.Argument;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.generic.ParameterDataType;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.resolver.ResolverExpression;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.resolver.ResolverReference;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.valueprovider.ValueProviderDefinition;
import com.mulesoft.connectivity.rest.sdk.templating.api.RestSdkRunConfiguration;
import com.mulesoft.connectivity.rest.sdk.templating.exception.TemplatingException;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.parameter.SdkField;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.resolver.AbstractSdkResolverProvider;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.util.SdkTemplatingUtils;

import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.LocalDateTime;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import com.google.common.base.CaseFormat;
import com.squareup.javapoet.AnnotationSpec;
import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.CodeBlock;
import com.squareup.javapoet.FieldSpec;
import com.squareup.javapoet.JavaFile;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.ParameterizedTypeName;
import com.squareup.javapoet.TypeSpec;

public abstract class JavaTemplateEntity extends TemplateEntity {

  protected static final String VALUE_MEMBER = SdkTemplatingUtils.VALUE_MEMBER;
  protected static final String NAME_MEMBER = "name";
  protected static final String BASE_PACKAGE_SUFFIX = ".base";
  protected static final String INTERCEPTOR_PACKAGE_SUFFIX = ".interceptor";
  protected static final String BASE_CLASSNAME_SUFFIX = "Base";
  protected static final String INTERCEPTOR_CLASSNAME_SUFFIX = "Interceptor";

  private static final String NULL = "null";

  protected final Path outputDir;
  protected final ConnectorModel connectorModel;
  protected final RestSdkRunConfiguration runConfiguration;

  public JavaTemplateEntity(Path outputDir, ConnectorModel connectorModel, RestSdkRunConfiguration runConfiguration) {
    this.outputDir = outputDir;
    this.connectorModel = connectorModel;
    this.runConfiguration = runConfiguration;
  }

  protected void writeClassToFile(TypeSpec clazz, String targetPackage) throws TemplatingException {
    writeClassToFile(clazz, targetPackage, false);
  }

  protected void writeClassToFile(TypeSpec clazz, String targetPackage, boolean isRefinement) throws TemplatingException {
    writeJavaFile(getJavaFileBuilderForClass(clazz, targetPackage).build(), isRefinement);
  }

  protected JavaFile.Builder getJavaFileBuilderForClass(TypeSpec clazz, String targetPackage) {
    return JavaFile
        .builder(targetPackage, clazz)
        .skipJavaLangImports(true);
  }

  protected void writeJavaFile(JavaFile javaFile) throws TemplatingException {
    writeJavaFile(javaFile, false);
  }

  protected void writeJavaFile(JavaFile javaFile, boolean isRefinement) throws TemplatingException {
    try {
      if (shouldGenerate(javaFile)) {
        javaFile.writeTo(outputDir.resolve(runConfiguration.generatedSourceDir()));
      } else if (!isRefinement) {
        final String packageDirectory = javaFile.packageName.replace('.', File.separatorChar);
        runConfiguration.messageCollector().addInfo("Ignoring: '" + packageDirectory + File.separatorChar + javaFile.typeSpec.name
            + ".java" + "' as it was overwritten.");
        final Path ignoredDir = outputDir.resolve(runConfiguration.ignoredDir());
        javaFile.writeTo(ignoredDir);
        Path generatedIgnoredFile = resolveJavaFile(javaFile, runConfiguration.ignoredDir());
        if (generatedIgnoredFile.toFile().exists()) {
          compareToOverrideOne(javaFile, generatedIgnoredFile, runConfiguration.overrideDir());
          compareToOverrideOne(javaFile, generatedIgnoredFile, runConfiguration.sourceDir());
        }
      }
    } catch (Exception e) {
      throw new TemplatingException("There was an error when writing " + this.getClass().getName() + " class", e);
    }
  }

  private void compareToOverrideOne(JavaFile javaFile, Path ignoredFile, String overrideDir) throws IOException {
    final Path overwrittenFile = resolveJavaFile(javaFile, overrideDir);
    if (overwrittenFile.toFile().exists()) {
      final String currentContent = new String(Files.readAllBytes(overwrittenFile), StandardCharsets.UTF_8);
      final String newContent = new String(Files.readAllBytes(ignoredFile), StandardCharsets.UTF_8);
      boolean isIdentical = toCanonicalWay(currentContent).equals(toCanonicalWay(newContent));
      if (isIdentical) {
        // this will also take care of spaces like tabs etc.
        runConfiguration.messageCollector()
            .addWarning("Overwritten file `" + overwrittenFile + "` can be removed as is the same as the generated one.");
      }
    }
  }

  private String toCanonicalWay(String currentContent) {
    return currentContent.replaceAll("\\s+", " ");
  }

  private boolean shouldGenerate(JavaFile javaFile) {
    boolean isOldWay = !runConfiguration.regenerateMode();
    if (isOldWay) {
      // This mean that we are still generating java classes in src/main/java and they are still being pushed
      return true;
    } else {
      final Path inOverride = resolveJavaFile(javaFile, runConfiguration.overrideDir());
      final Path inSourceDir = resolveJavaFile(javaFile, runConfiguration.sourceDir());
      return !inOverride.toFile().exists() && !inSourceDir.toFile().exists();
    }
  }

  private Path resolveJavaFile(JavaFile javaFile, String targetRelativeFolderPath) {
    final String javaFileName = javaFile.typeSpec.name + ".java";
    final String packageDirectory = javaFile.packageName.replace('.', File.separatorChar);
    final Path resolve = outputDir.resolve(targetRelativeFolderPath)
        .resolve(packageDirectory)
        .resolve(javaFileName);
    return resolve;
  }

  public Path getResourcesPath() {
    return outputDir.resolve(runConfiguration.generatedResourcesDir());
  }

  public Path getSourcesPath() {
    return outputDir.resolve("src");
  }

  protected MethodSpec.Builder generateOptionalGetter(FieldSpec fieldSpec, Class<?> type, CaseFormat fieldNameCaseFormat) {
    String name = "get" + fieldNameCaseFormat.to(UPPER_CAMEL, fieldSpec.name);

    return MethodSpec
        .methodBuilder(name)
        .addModifiers(PUBLIC)
        .returns(ParameterizedTypeName.get(Optional.class, type))
        .addCode(CodeBlock.builder().addStatement("return $T.ofNullable(this.$N)", Optional.class, fieldSpec).build());
  }

  public static FieldSpec getConstantStringField(String fieldName, String value) {
    FieldSpec.Builder fieldSpecBuilder = FieldSpec
        .builder(String.class, fieldName, PRIVATE, STATIC, FINAL);

    if (isNotBlank(value)) {
      fieldSpecBuilder.initializer("$S", value);
    } else {
      fieldSpecBuilder.initializer("$L", NULL);
    }

    return fieldSpecBuilder.build();
  }

  protected Class<?> getJavaType(ParameterDataType parameterDataType) {
    if (parameterDataType.equals(ParameterDataType.LOCAL_DATE_TIME)) {
      return LocalDateTime.class;
    } else if (parameterDataType.equals(ParameterDataType.ZONED_DATE_TIME)) {
      return ZonedDateTime.class;
    } else if (parameterDataType.equals(ParameterDataType.STRING)) {
      return String.class;
    } else if (parameterDataType.equals(ParameterDataType.INTEGER)) {
      return Integer.class;
    } else if (parameterDataType.equals(ParameterDataType.LONG)) {
      return Long.class;
    } else if (parameterDataType.equals(ParameterDataType.NUMBER)) {
      return Double.class;
    } else if (parameterDataType.equals(ParameterDataType.BOOLEAN)) {
      return boolean.class;
    }

    throw new IllegalArgumentException(parameterDataType.getName() + " is not supported");
  }

  protected AnnotationSpec getValueProviderAnnotation(SdkField sdkField, String contentParameterName) {
    AbstractSdkResolverProvider valueProvider = sdkField.getSdkValueProvider();
    AnnotationSpec.Builder annotationBuilder = AnnotationSpec
        .builder(FieldValues.class)
        .addMember("value", "$T.class",
                   ClassName.get(valueProvider.getPackage(), valueProvider.getJavaClassName()))
        .addMember("targetSelectors", "$S", sdkField.getExternalName());

    List<Argument> arguments = getArgumentsFromValueProvider(sdkField.getField().getValueProvider());
    for (Argument argument : arguments) {
      annotationBuilder.addMember("bindings", "$L",
                                  AnnotationSpec.builder(Binding.class)
                                      .addMember("actingParameter", "$S", getActingParameterJavaName(argument))
                                      .addMember("extractionExpression", "$S",
                                                 transformBodyPrefix(argument.getValue().getValue(), contentParameterName))
                                      .build());
    }
    return annotationBuilder.build();
  }

  protected List<Argument> getArgumentsFromValueProvider(ResolverExpression<ValueProviderDefinition> valueProviderExpression) {

    if (valueProviderExpression instanceof ValueProviderDefinition) {
      ValueProviderDefinition valueProviderDefinition = (ValueProviderDefinition) valueProviderExpression;
      return getArgumentsFromValueProviderDefinition(valueProviderDefinition);
    } else if (valueProviderExpression instanceof ResolverReference) {
      ResolverReference<ValueProviderDefinition> valueProviderReference =
          (ResolverReference<ValueProviderDefinition>) valueProviderExpression;

      return valueProviderReference.getArguments();
    }

    throw new IllegalArgumentException("Invalid valueProviderExpression. This is a bug.");
  }

  protected List<Argument> getArgumentsFromValueProviderDefinition(ValueProviderDefinition valueProviderDefinition) {
    if (valueProviderDefinition.getRequest() == null) {
      return emptyList();
    }
    HttpRequestBinding httpRequestBinding = valueProviderDefinition.getRequest().getHttpRequestBinding();
    List<Argument> arguments = new ArrayList<>();
    if (httpRequestBinding != null) {
      arguments.addAll(httpRequestBinding.getQueryParameter());
      arguments.addAll(httpRequestBinding.getUriParameter());
      arguments.addAll(httpRequestBinding.getHeader());
    }

    return arguments.stream()
        .filter(argument -> isBodyBindingUsed(argument))
        .collect(Collectors.toList());
  }
}
