/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.templating.sdk;

import static com.google.common.base.CaseFormat.LOWER_HYPHEN;
import static com.google.common.base.CaseFormat.UPPER_CAMEL;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.util.JavaUtils.getJavaUpperCamelNameFromXml;
import static com.mulesoft.connectivity.rest.sdk.internal.webapi.util.ParserUtils.splitCaps;
import static java.util.stream.Collectors.toList;
import static javax.lang.model.element.Modifier.PUBLIC;

import org.mule.runtime.api.meta.Category;
import org.mule.runtime.extension.api.annotation.Configurations;
import org.mule.runtime.extension.api.annotation.Extension;
import org.mule.runtime.extension.api.annotation.dsl.xml.Xml;
import org.mule.runtime.extension.api.annotation.error.ErrorTypes;
import org.mule.runtime.extension.api.annotation.license.RequiresEnterpriseLicense;

import com.mulesoft.connectivity.rest.commons.api.error.RestError;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.schema.TypeSchema;
import com.mulesoft.connectivity.rest.sdk.templating.JavaTemplateEntity;
import com.mulesoft.connectivity.rest.sdk.templating.api.RestSdkRunConfiguration;
import com.mulesoft.connectivity.rest.sdk.templating.exception.TemplatingException;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.configuration.SdkConfig;

import java.nio.file.Path;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Stream;

import com.squareup.javapoet.AnnotationSpec;
import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.CodeBlock;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.TypeSpec;

public class SdkConnector extends JavaTemplateEntity {

  private final List<SdkConfig> configs;
  private final Map<TypeSchema, String> typeSchemaNames = new HashMap<>();
  private final Map<String, TypeName> parentValueProviders = new HashMap<>();
  private final Map<String, TypeName> parentSampleDataProviders = new HashMap<>();

  public SdkConnector(Path outputDir, ConnectorModel connectorModel, RestSdkRunConfiguration runConfiguration)
      throws TemplatingException {
    super(outputDir, connectorModel, runConfiguration);

    this.configs = Stream.of(new SdkConfig(outputDir, connectorModel, this, runConfiguration)).collect(toList());
  }

  public String getJavaName() {
    return getJavaUpperCamelNameFromXml(connectorModel.getConnectorXmlName());
  }

  private Category getCategory() {
    return connectorModel.getCategory() != null ? Category.valueOf(connectorModel.getCategory().getCategoryName().toUpperCase())
        : null;
  }

  @Override
  public void applyTemplates() throws TemplatingException {
    generateConnectorClass();

    for (SdkConfig config : configs) {
      config.applyTemplates();
    }
  }

  private void generateConnectorClass() throws TemplatingException {

    TypeSpec.Builder connectorClassBuilder =
        TypeSpec
            .classBuilder(getJavaUpperCamelNameFromXml(connectorModel.getConnectorXmlName()))
            .addModifiers(PUBLIC)
            .addAnnotation(getExtensionAnnotation())
            .addAnnotation(getXmlAnnotation())
            .addAnnotation(getConfigurationsAnnotation())
            .addAnnotation(getErrorTypesAnnotation());

    if (!Category.COMMUNITY.equals(getCategory())) {
      connectorClassBuilder.addAnnotation(getLicenseAnnotation());
    }

    writeClassToFile(connectorClassBuilder.build(), connectorModel.getBasePackage() + ".internal.extension");
  }

  private AnnotationSpec getExtensionAnnotation() {
    String extensionName = LOWER_HYPHEN.to(UPPER_CAMEL, connectorModel.getConnectorXmlName());
    extensionName = splitCaps(extensionName, " ");

    AnnotationSpec.Builder extensionAnnotationBuilder = AnnotationSpec.builder(Extension.class)
        .addMember(NAME_MEMBER, "$S", extensionName);

    if (getCategory() != null) {
      extensionAnnotationBuilder.addMember("category", "$T.$L", Category.class, getCategory());
    }

    return extensionAnnotationBuilder.build();
  }

  private AnnotationSpec getLicenseAnnotation() {
    AnnotationSpec.Builder connectorAnnotationBuilder = AnnotationSpec.builder(RequiresEnterpriseLicense.class)
        .addMember("allowEvaluationLicense", "$L", true);

    return connectorAnnotationBuilder.build();
  }

  private AnnotationSpec getXmlAnnotation() {
    AnnotationSpec.Builder connectorAnnotationBuilder = AnnotationSpec.builder(Xml.class)
        .addMember("prefix", "$S", connectorModel.getExtensionXml());

    return connectorAnnotationBuilder.build();
  }

  private AnnotationSpec getConfigurationsAnnotation() {
    CodeBlock.Builder codeBlock = CodeBlock.builder();

    codeBlock.add("{");

    for (int i = 0; i < this.configs.size(); i++) {
      SdkConfig config = this.configs.get(i);
      codeBlock.add("$T.class", config.getTypeName());
      if (i < this.configs.size() - 1) {
        codeBlock.add(",");
      }
    }

    codeBlock.add("}");

    AnnotationSpec.Builder configurationsAnnotationBuilder =
        AnnotationSpec
            .builder(Configurations.class)
            .addMember(VALUE_MEMBER, codeBlock.build());

    return configurationsAnnotationBuilder.build();
  }

  private AnnotationSpec getErrorTypesAnnotation() {
    return AnnotationSpec
        .builder(ErrorTypes.class)
        .addMember(VALUE_MEMBER, "$T.class", RestError.class)
        .build();
  }

  public Map<TypeSchema, String> getTypeSchemaNames() {
    return typeSchemaNames;
  }

  public Map<String, TypeName> getParentValueProviders() {
    return parentValueProviders;
  }

  public Map<String, TypeName> getParentSampleDataProviders() {
    return parentSampleDataProviders;
  }
}
