/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.templating.sdk.connection.authentication;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.ConnectorSecurityScheme.SecuritySchemeType.BASIC;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.ConnectorSecurityScheme.SecuritySchemeType.BEARER;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.ConnectorSecurityScheme.SecuritySchemeType.DIGEST_AUTHENTICATION;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.ConnectorSecurityScheme.SecuritySchemeType.OAUTH2;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.ConnectorSecurityScheme.SecuritySchemeType.PASS_THROUGH;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.ConnectorSecurityScheme.SecuritySchemeType.UNSECURED;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.OAuth2Scheme.GrantType.AUTHORIZATION_CODE;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.OAuth2Scheme.GrantType.CLIENT_CREDENTIALS;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.ConnectorSecurityScheme;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.OAuth2Scheme;
import com.mulesoft.connectivity.rest.sdk.templating.exception.TemplatingException;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.parameter.SdkParameter;

import com.squareup.javapoet.FieldSpec;
import com.squareup.javapoet.TypeSpec;

public abstract class SdkAuthenticationStrategy {

  public static SdkAuthenticationStrategy getStrategy(ConnectorSecurityScheme securityScheme) throws TemplatingException {
    ConnectorSecurityScheme.SecuritySchemeType securitySchemeType = securityScheme.getSchemeType();

    if (securitySchemeType.equals(BASIC)) {
      return new SdkBasicAuthenticationStrategy(securityScheme);
    } else if (securitySchemeType.equals(DIGEST_AUTHENTICATION)) {
      return new SdkDigestAuthenticationStrategy(securityScheme);
    } else if (securitySchemeType.equals(PASS_THROUGH)) {
      return new SdkBaseAuthenticationStrategy(securityScheme);
    } else if (securitySchemeType.equals(UNSECURED)) {
      return new SdkBaseAuthenticationStrategy(securityScheme);
    } else if (securitySchemeType.equals(OAUTH2)) {
      if (((OAuth2Scheme) securityScheme).getGrantType().equals(AUTHORIZATION_CODE)) {
        return new SdkAuthorizationCodeAuthenticationStrategy(securityScheme);
      } else if (((OAuth2Scheme) securityScheme).getGrantType().equals(CLIENT_CREDENTIALS)) {
        return new SdkClientCredentialsAuthenticationStrategy(securityScheme);
      }
    } else if (securitySchemeType.equals(BEARER)) {
      return new SdkBearerAuthenticationStrategy(securityScheme);
    }

    throw new TemplatingException("No Strategy available for '" + securitySchemeType + "'");
  }

  protected final ConnectorSecurityScheme securityScheme;

  protected SdkAuthenticationStrategy(ConnectorSecurityScheme securityScheme) {
    this.securityScheme = securityScheme;
  }

  public abstract Class<?> getCommonsBaseClass();

  public abstract void addClassAnnotations(TypeSpec.Builder connectionProviderClassBuilder);

  public abstract void addDefaultClassMembers(TypeSpec.Builder connectionProviderClassBuilder);

  public boolean shouldTestConnectivity() {
    return securityScheme.getTestConnectionConfig() != null;
  }

  public FieldSpec.Builder getParameterField(SdkParameter parameter) {
    return parameter.generateParameterField();
  }

  public boolean canOverrideCreateConnectionMethod() {
    return true;
  }
}
