/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.templating.sdk.parameter;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.Parameter;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.resolver.ResolverDefinition;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.resolver.ResolverReference;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.ArrayTypeDefinition;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.TypeDefinition;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.valueprovider.ValueProviderDefinition;
import com.mulesoft.connectivity.rest.sdk.templating.JavaTemplateEntity;
import com.mulesoft.connectivity.rest.sdk.templating.api.RestSdkRunConfiguration;
import com.mulesoft.connectivity.rest.sdk.templating.exception.TemplatingException;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.SdkConnector;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.operation.SdkOperation;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.resolver.AbstractSdkResolverProvider;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.util.TypeDefinitionUtil;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.valueprovider.SdkValueProviderDefinition;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.valueprovider.SdkValueProviderInline;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.valueprovider.SdkValueProviderReference;
import com.squareup.javapoet.AnnotationSpec;
import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.CodeBlock;
import com.squareup.javapoet.FieldSpec;
import com.squareup.javapoet.JavaFile;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.ParameterSpec;
import com.squareup.javapoet.ParameterizedTypeName;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.TypeSpec;
import org.mule.runtime.extension.api.annotation.Alias;
import org.mule.runtime.extension.api.annotation.connectivity.oauth.OAuthParameter;
import org.mule.runtime.extension.api.annotation.param.NullSafe;
import org.mule.runtime.extension.api.annotation.param.display.DisplayName;
import org.mule.runtime.extension.api.annotation.param.display.Summary;
import org.mule.runtime.extension.api.annotation.values.OfValues;

import java.lang.reflect.Type;
import java.nio.file.Path;
import java.util.Date;
import java.util.List;
import java.util.Optional;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.util.JavaUtils.abbreviateText;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.util.JavaUtils.getJavaConstantNameFromXml;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.util.JavaUtils.getParameterJavaName;
import static com.mulesoft.connectivity.rest.sdk.internal.webapi.util.XmlUtils.getXmlName;
import static com.squareup.javapoet.MethodSpec.constructorBuilder;
import static com.squareup.javapoet.MethodSpec.methodBuilder;
import static com.squareup.javapoet.TypeSpec.anonymousClassBuilder;
import static java.util.Optional.ofNullable;
import static javax.lang.model.element.Modifier.PRIVATE;
import static javax.lang.model.element.Modifier.PUBLIC;
import static org.apache.commons.lang3.StringUtils.capitalize;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

public class SdkParameter extends JavaTemplateEntity {

  protected final Parameter parameter;
  protected final SdkConnector connector;
  protected final JavaTemplateEntity parentElement;
  private final String parentJavaName;

  private AbstractSdkResolverProvider sdkValueProvider;

  public SdkParameter(Path outputDir, ConnectorModel connectorModel, SdkConnector sdkConnector, String parentJavaName,
                      Parameter parameter, JavaTemplateEntity parentElement, RestSdkRunConfiguration runConfiguration) {
    super(outputDir, connectorModel, runConfiguration);
    this.parameter = parameter;
    this.connector = sdkConnector;
    this.parentJavaName = parentJavaName;
    this.parentElement = parentElement;
  }

  public String getDescription() {
    return parameter.getDescription();
  }

  public String getDisplayName() {
    return parameter.getDisplayName();
  }

  protected String getInternalName() {
    return parameter.getInternalName();
  }

  public String getMuleAlias() {
    return parameter.getMuleAlias();
  }

  public String getExternalName() {
    return parameter.getExternalName();
  }

  protected TypeDefinition getTypeDefinition() {
    return parameter.getTypeDefinition();
  }

  protected String getDefaultValue() {
    return parameter.getDefaultValue();
  }

  protected boolean getRequired() {
    return parameter.isRequired();
  }

  public ParameterType getParameterType() {
    return parameter.getParameterType();
  }

  protected String getEnumClassName() {
    return isEnum() || isInnerTypeEnum() ? buildEnumName(parentJavaName, this)
        : null;
  }

  protected String getEnumClassPackage() {
    return connectorModel.getBasePackage() + ".api.metadata";
  }

  protected AbstractSdkResolverProvider getSdkValueProvider() {
    if (this.sdkValueProvider != null) {
      return sdkValueProvider;
    }

    if (parentElement instanceof SdkOperation) {
      if (parameter.getValueProvider() != null) {
        if (parameter.getValueProvider() instanceof ResolverDefinition) {
          sdkValueProvider = new SdkValueProviderInline(outputDir,
                                                        connectorModel,
                                                        (SdkOperation) parentElement,
                                                        parameter.getInternalName(),
                                                        new SdkValueProviderDefinition(parameter.getValueProvider()),
                                                        runConfiguration,
                                                        true);

          return sdkValueProvider;

        } else if (parameter.getValueProvider() instanceof ResolverReference) {
          sdkValueProvider = new SdkValueProviderReference(outputDir,
                                                           connectorModel,
                                                           connector,
                                                           (SdkOperation) parentElement,
                                                           parameter.getInternalName(),
                                                           (ResolverReference<ValueProviderDefinition>) parameter
                                                               .getValueProvider(),
                                                           new SdkValueProviderDefinition(parameter.getValueProvider()),
                                                           runConfiguration,
                                                           true);

          return sdkValueProvider;
        }

        throw new IllegalArgumentException("Value Provider generation not supported: " + parameter.getValueProvider());
      }
    }

    return null;
  }

  public String getJavaName() {
    return getParameterJavaName(getInternalName(), isArrayType());
  }

  public boolean isArrayType() {
    return getTypeDefinition() instanceof ArrayTypeDefinition;
  }

  public TypeName getTypeName() {
    return getParameterTypeName(getTypeDefinition(), false);
  }

  private TypeName getParameterTypeName(TypeDefinition typeDefinition, boolean isInnerType) {
    if (typeDefinition instanceof ArrayTypeDefinition) {
      TypeDefinition innerType = ((ArrayTypeDefinition) typeDefinition).getInnerType();

      return ParameterizedTypeName.get(ClassName.get(List.class), getParameterTypeName(innerType, true));
    } else if (typeDefinition.isEnum()) {
      return ClassName.get(getEnumClassPackage(), getEnumClassName());
    } else {
      Type parameterPrimitiveJavaType = getParameterPrimitiveJavaType(typeDefinition);
      if (isInnerType && parameterPrimitiveJavaType.equals(boolean.class)) {
        return TypeName.get(Boolean.class);
      } else {
        return TypeName.get(parameterPrimitiveJavaType);
      }
    }
  }

  private static Type getParameterPrimitiveJavaType(TypeDefinition typeDefinition) {
    if (typeDefinition instanceof ArrayTypeDefinition) {
      // Should not reach here. Array Types should not depend on this method (Only for it's inner type).
      // Array's will throw this exception as they are supported for query/uri/header parameters.
      throw new IllegalArgumentException("Type is not primitive. Should handle arrays in templating.");
    }

    Type javaType = TypeDefinitionUtil.getJavaType(typeDefinition);

    if (javaType.equals(Date.class)) {
      return String.class;
    }

    return javaType;
  }

  @Override
  public void applyTemplates() throws TemplatingException {
    if (isEnum() || isInnerTypeEnum()) {
      generateEnumClass();
    }

    if (getSdkValueProvider() != null) {
      getSdkValueProvider().applyTemplates();
    }
  }

  private void generateEnumClass() throws TemplatingException {
    MethodSpec enumConstructor = constructorBuilder()
        .addParameter(String.class, "value")
        .addCode(CodeBlock.builder().addStatement("this.value = value").build())
        .build();

    MethodSpec valueGetterMethod = methodBuilder("getValue")
        .addModifiers(PUBLIC)
        .returns(String.class)
        .addCode(CodeBlock.builder().addStatement("return value").build())
        .build();

    TypeSpec.Builder enumClassBuilder = TypeSpec.enumBuilder(getEnumClassName())
        .addModifiers(PUBLIC)
        .addField(String.class, "value", PRIVATE)
        .addMethod(enumConstructor)
        .addMethod(valueGetterMethod);

    for (String enumConstant : getEnumValues()) {
      enumClassBuilder.addEnumConstant(getJavaConstantNameFromXml(getXmlName(enumConstant)),
                                       anonymousClassBuilder("$S", enumConstant).build());
    }

    JavaFile.Builder javaFileBuilder = JavaFile
        .builder(getEnumClassPackage(), enumClassBuilder.build())
        .skipJavaLangImports(true);

    writeJavaFile(javaFileBuilder.build());
  }

  private static String buildEnumName(String parentJavaName,
                                      SdkParameter sdkParameter) {

    String parameterNameBasedEnumName = capitalize(sdkParameter.getJavaName()) + "Enum";
    // Use operation name + parameter name to generate the enum name
    String operationNameBasedEnumName = parentJavaName + parameterNameBasedEnumName;
    return operationNameBasedEnumName;
  }

  public ParameterSpec.Builder generateParameterParameter(boolean withAnnotation) {
    return generateParameterParameter(null, withAnnotation);
  }

  public ParameterSpec.Builder generateParameterParameter(TypeName forcedTypeName, boolean withAnnotation) {

    ParameterSpec.Builder paramSpecBuilder;
    if (forcedTypeName == null) {
      paramSpecBuilder = ParameterSpec.builder(getTypeName(), getJavaName());
    } else {
      paramSpecBuilder = ParameterSpec.builder(forcedTypeName, getJavaName());
    }

    if (withAnnotation) {
      AnnotationSpec optionalAnnotation = getOptionalAnnotation();
      if (optionalAnnotation != null) {
        paramSpecBuilder.addAnnotation(optionalAnnotation);
      }

      getNullSafeAnnotation().ifPresent(paramSpecBuilder::addAnnotation);

      paramSpecBuilder.addAnnotation(getDisplayNameAnnotation());

      getMuleAliasAnnotation().ifPresent(paramSpecBuilder::addAnnotation);
      getSummaryAnnotation().ifPresent((paramSpecBuilder::addAnnotation));

      if (getSdkValueProvider() != null) {
        paramSpecBuilder.addAnnotation(getValueProviderAnnotation());
      }

      configureParameterSpecBuilder(paramSpecBuilder);
    }

    return paramSpecBuilder;
  }

  protected void configureParameterSpecBuilder(ParameterSpec.Builder paramSpecBuilder) {}

  private AnnotationSpec getValueProviderAnnotation() {
    return AnnotationSpec
        .builder(OfValues.class)
        .addMember("value", "$T.class",
                   ClassName.get(getSdkValueProvider().getPackage(), getSdkValueProvider().getJavaClassName()))
        .build();
  }

  public FieldSpec.Builder generateParameterField() {
    AnnotationSpec parameterAnnotation = AnnotationSpec
        .builder(org.mule.runtime.extension.api.annotation.param.Parameter.class)
        .build();

    return generateParameterField(parameterAnnotation);
  }

  public FieldSpec.Builder generateOAuthParameterField() {
    AnnotationSpec parameterAnnotation = AnnotationSpec
        .builder(OAuthParameter.class)
        .addMember("requestAlias", "$S", getExternalName())
        .build();

    return generateParameterField(parameterAnnotation);
  }

  private FieldSpec.Builder generateParameterField(AnnotationSpec parameterAnnotation) {
    FieldSpec.Builder fieldSpecBuilder = FieldSpec
        .builder(getTypeName(), getJavaName())
        .addAnnotation(parameterAnnotation)
        .addAnnotation(getDisplayNameAnnotation());

    AnnotationSpec optionalAnnotation = getOptionalAnnotation();
    if (optionalAnnotation != null) {
      fieldSpecBuilder.addAnnotation(optionalAnnotation);
    }
    getMuleAliasAnnotation().ifPresent(fieldSpecBuilder::addAnnotation);

    getSummaryAnnotation().ifPresent(fieldSpecBuilder::addAnnotation);

    return fieldSpecBuilder;
  }

  private Optional<AnnotationSpec> getNullSafeAnnotation() {
    return ofNullable((!getRequired() && isArrayType()) ? AnnotationSpec.builder(NullSafe.class).build() : null);
  }

  private AnnotationSpec getOptionalAnnotation() {
    if (!getRequired() || isNotBlank(getDefaultValue())) {
      AnnotationSpec.Builder optionalBuilder =
          AnnotationSpec.builder(org.mule.runtime.extension.api.annotation.param.Optional.class);

      if (isNotBlank(getDefaultValue())) {
        if (isEnum()) {
          optionalBuilder.addMember("defaultValue", "$S",
                                    getJavaConstantNameFromXml(getXmlName(getDefaultValue())));
        } else {
          optionalBuilder.addMember("defaultValue", "$S", getDefaultValue());
        }
      }

      return optionalBuilder.build();
    }

    return null;
  }

  private AnnotationSpec getDisplayNameAnnotation() {
    return AnnotationSpec
        .builder(DisplayName.class)
        .addMember(VALUE_MEMBER, "$S", getDisplayName())
        .build();
  }

  private Optional<AnnotationSpec> getMuleAliasAnnotation() {
    return ofNullable(isNotBlank(getMuleAlias()) ? AnnotationSpec
        .builder(Alias.class)
        .addMember(VALUE_MEMBER, "$S", getMuleAlias())
        .build() : null);
  }

  private Optional<AnnotationSpec> getSummaryAnnotation() {
    return ofNullable(isNotBlank(getDescription()) ? AnnotationSpec
        .builder(Summary.class)
        .addMember(VALUE_MEMBER, "$S", abbreviateText(getDescription()))
        .build() : null);
  }

  public CodeBlock getStringValueGetter() {
    return getStringValueGetter(getTypeDefinition(), getJavaName(), getRequired());
  }

  public CodeBlock getInnerTypeStringValueGetter(String varName) {
    return getStringValueGetter(((ArrayTypeDefinition) getTypeDefinition()).getInnerType(), varName, true);
  }

  private static CodeBlock getStringValueGetter(TypeDefinition typeDefinition, String varName, boolean required) {
    CodeBlock.Builder blockBuilder = CodeBlock.builder();
    Type javaType = TypeDefinitionUtil.getJavaType(typeDefinition);

    boolean requiresNullCheck = requiresNullCheck(required, typeDefinition.isEnum(), javaType);

    if (requiresNullCheck) {
      blockBuilder.add("$L != null ? ", varName);
    }

    if (typeDefinition.isEnum()) {
      blockBuilder.add("$L.getValue()", varName);
    } else if (javaType.equals(String.class)) {
      blockBuilder.add(varName);
    } else {
      blockBuilder.add("$L", varName);
    }

    if (requiresNullCheck) {
      blockBuilder.add(" : null");
    }

    return blockBuilder.build();
  }

  private static boolean requiresNullCheck(boolean required, boolean isEnum, Type javaType) {
    return !required && (isEnum || (!javaType.equals(boolean.class) && !javaType.equals(String.class)));
  }

  private boolean isEnum() {
    return getTypeDefinition().isEnum();
  }

  private boolean isInnerTypeEnum() {
    return getTypeDefinition() instanceof ArrayTypeDefinition
        && ((ArrayTypeDefinition) getTypeDefinition()).getInnerType().isEnum();
  }

  private List<String> getEnumValues() {
    if (isEnum()) {
      return getTypeDefinition().getEnumValues();
    } else if (isInnerTypeEnum()) {
      return ((ArrayTypeDefinition) getTypeDefinition()).getInnerType().getEnumValues();
    } else {
      return null;
    }
  }
}
