/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.templating.sdk.trigger.layers;

import static com.mulesoft.connectivity.rest.sdk.templating.sdk.sampledata.SdkSampleDataFactory.getSdkSampleDataResolver;
import static javax.lang.model.element.Modifier.PUBLIC;
import static org.apache.commons.lang3.StringUtils.defaultIfEmpty;

import org.mule.runtime.extension.api.annotation.metadata.MetadataScope;
import org.mule.runtime.extension.api.annotation.param.MediaType;
import org.mule.runtime.extension.api.annotation.param.display.DisplayName;
import org.mule.runtime.extension.api.annotation.Alias;
import org.mule.sdk.api.annotation.Ignore;
import org.mule.sdk.api.annotation.data.sample.SampleData;

import com.mulesoft.connectivity.rest.commons.api.datasense.metadata.output.SdkHttpResponseAttributesMetadataResolver;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.trigger.Trigger;
import com.mulesoft.connectivity.rest.sdk.templating.api.RestSdkRunConfiguration;
import com.mulesoft.connectivity.rest.sdk.templating.exception.TemplatingException;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.SdkConnector;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.metadata.SdkTriggerOutputMetadataResolver;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.parameter.SdkParameter;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.resolver.AbstractSdkResolverProvider;

import java.nio.file.Path;
import java.util.List;

import com.squareup.javapoet.AnnotationSpec;
import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.CodeBlock;
import com.squareup.javapoet.JavaFile;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.TypeSpec;

public class SdkTriggerImplementationLayer extends AbstractSdkTriggerLayer {

  public static final String PARAM_DOC_NAME_DESCRIPTION = "@param $L $L\n";

  private final TypeName superclass;
  private final SdkTriggerOutputMetadataResolver outputMetadataResolver;

  private final List<SdkParameter> sdkParameters;

  private AbstractSdkResolverProvider sdkSampleDataProvider;

  public SdkTriggerImplementationLayer(Path outputDir, ConnectorModel connectorModel, SdkConnector sdkConnector, Trigger trigger,
                                       String javaClassName, String packageName, TypeName superclass,
                                       List<SdkParameter> sdkParameters,
                                       RestSdkRunConfiguration runConfiguration)
      throws TemplatingException {
    super(outputDir, connectorModel, sdkConnector, trigger, javaClassName, packageName, runConfiguration);
    this.superclass = superclass;
    this.outputMetadataResolver = buildOutputMetadataResolver(outputDir, connectorModel, sdkConnector, trigger, runConfiguration);
    this.sdkParameters = sdkParameters;
  }

  private SdkTriggerOutputMetadataResolver buildOutputMetadataResolver(Path outputDir, ConnectorModel connectorModel,
                                                                       SdkConnector sdkConnector, Trigger trigger,
                                                                       RestSdkRunConfiguration runConfiguration)
      throws TemplatingException {

    return new SdkTriggerOutputMetadataResolver(outputDir, connectorModel, sdkConnector, trigger.getInternalName(),
                                                trigger.getTriggerOutputType(), trigger.getName(), runConfiguration);
  }

  private void generateTriggerClass() throws TemplatingException {
    CodeBlock.Builder javaDoc = CodeBlock.builder()
        .add("\n$L\n", defaultIfEmpty(trigger.getDescription(), trigger.getDisplayName()))
        .add("\n");

    for (SdkParameter parameter : sdkParameters) {
      javaDoc.add(PARAM_DOC_NAME_DESCRIPTION, parameter.getExternalName(),
                  defaultIfEmpty(parameter.getDescription(), parameter.getDisplayName()));
    }

    TypeSpec.Builder triggerClassBuilder =
        TypeSpec
            .classBuilder(getJavaClassName())
            .addModifiers(PUBLIC)
            .superclass(superclass)
            .addAnnotation(generateMediaTypeAnnotation())
            .addAnnotation(generateDisplayNameAnnotation())
            .addJavadoc(javaDoc.build());

    if (trigger.getAlias() != null) {
      triggerClassBuilder.addAnnotation(generateAliasAnnotation());
    }

    sdkSampleDataProvider = buildSampleData();
    if (sdkSampleDataProvider != null) {
      addSampleDataAnnotation(triggerClassBuilder);
    }



    triggerClassBuilder.addAnnotation(getMetadataScopeAnnotation());

    if (trigger.isSidecar()) {
      triggerClassBuilder.addAnnotation(generateSidecarAnnotation());
    }

    JavaFile.Builder javaFileBuilder = getJavaFileBuilderForClass(triggerClassBuilder.build(), getPackage());
    writeJavaFile(javaFileBuilder.build());
  }

  private AnnotationSpec generateMediaTypeAnnotation() {
    return AnnotationSpec
        .builder(MediaType.class)
        .addMember(VALUE_MEMBER, "$S", trigger.getTriggerOutputType().getMediaType())
        .build();
  }

  private AnnotationSpec getMetadataScopeAnnotation() {
    AnnotationSpec.Builder builder = AnnotationSpec
        .builder(MetadataScope.class);

    if (outputMetadataResolver != null && outputMetadataResolver.getRequiresMetadataResolver()) {
      builder
          .addMember("outputResolver", "$T.class",
                     ClassName.get(outputMetadataResolver.getPackage(),
                                   outputMetadataResolver.getClassName()));
    }

    return builder
        .addMember("attributesResolver", "$T.class", ClassName.get(SdkHttpResponseAttributesMetadataResolver.class))
        .build();
  }

  private AnnotationSpec generateSidecarAnnotation() {
    return AnnotationSpec.builder(Ignore.class)
        .build();
  }

  private AnnotationSpec generateDisplayNameAnnotation() {
    return AnnotationSpec.builder(DisplayName.class)
        .addMember(VALUE_MEMBER, "$S", trigger.getDisplayName())
        .build();
  }

  private AnnotationSpec generateAliasAnnotation() {
    return AnnotationSpec.builder(Alias.class)
        .addMember(VALUE_MEMBER, "$S", trigger.getAlias())
        .build();
  }

  private AbstractSdkResolverProvider buildSampleData() {
    return getSdkSampleDataResolver(trigger, connectorModel, sdkConnector, getJavaClassName(), sdkParameters, outputDir,
                                    runConfiguration);
  }

  private void addSampleDataAnnotation(TypeSpec.Builder triggerClassBuilder) {
    if (sdkSampleDataProvider != null) {
      AnnotationSpec sampleDataAnnotation = AnnotationSpec
          .builder(SampleData.class)
          .addMember(VALUE_MEMBER, "$T.class",
                     ClassName.get(sdkSampleDataProvider.getPackage(), sdkSampleDataProvider.getJavaClassName()))
          .build();

      triggerClassBuilder.addAnnotation(sampleDataAnnotation);
    }
  }

  @Override
  public void applyTemplates() throws TemplatingException {
    generateTriggerClass();

    if (outputMetadataResolver != null) {
      outputMetadataResolver.applyTemplates();
    }

    if (sdkSampleDataProvider != null) {
      sdkSampleDataProvider.applyTemplates();
    }
  }
}
