/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.templating.sdk;

import static java.lang.String.format;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.templating.VelocityTemplateEntity;
import com.mulesoft.connectivity.rest.sdk.templating.exception.TemplatingException;

import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Path;
import java.util.Properties;

public class SdkPom extends VelocityTemplateEntity {

  private final static String POM_VM = "templates/sdk/pom.vm";
  private static final String PROPERTIES_FROM_POM_PROPERTIES = "properties-from-pom.properties";
  private static final String PROJECT_VERSION = "project_version";

  private final ConnectorModel model;
  private final String restSdkProjectVersion;

  public SdkPom(Path outputDir, ConnectorModel model) throws TemplatingException {
    super(outputDir);
    this.model = model;
    this.restSdkProjectVersion = buildProjectVersion();
  }

  public String getName() {
    return model.getConnectorName();
  }

  public String getDescription() {
    return model.getApiDescription();
  }

  public String getMvnGroupId() {
    return model.getMvnGroupId();
  }

  public String getMvnArtifactId() {
    return isNotBlank(model.getMvnArtifactId()) ? model.getMvnArtifactId() : model.getConnectorXmlName();
  }

  public String getMvnVersion() {
    return model.getMvnVersion();
  }

  public String getRestSdkProjectVersion() {
    return restSdkProjectVersion;
  }

  @Override
  public void applyTemplates() throws TemplatingException {
    applyTemplate(POM_VM, outputDir.resolve("pom.xml"));
  }

  private String buildProjectVersion() throws TemplatingException {
    try (InputStream input = getClass().getClassLoader().getResourceAsStream(PROPERTIES_FROM_POM_PROPERTIES)) {
      if (input == null) {
        throw new TemplatingException(format("property file '%s' not found in the classpath. This is a bug.",
                                             PROPERTIES_FROM_POM_PROPERTIES));
      }
      Properties prop = new Properties();
      prop.load(input);
      String project_version = prop.getProperty(PROJECT_VERSION);

      if (isBlank(project_version)) {
        throw new TemplatingException(format("the property 'PROJECT_VERSION' (under '%s') is either null or empty. This is a bug.",
                                             PROPERTIES_FROM_POM_PROPERTIES));
      }

      return project_version;
    } catch (IOException e) {
      throw new TemplatingException(format("property file '%s' from the classpath could not be loaded. This is a bug.",
                                           PROPERTIES_FROM_POM_PROPERTIES),
                                    e);
    }
  }
}
